package org.sc3d.apt.jrider.v1;

/** Represents a complete car, consisting of five RigidBodies: four wheels and a chassis. This class knows how to render the Car using Faces, how to approximate it using CollisionSpheres, and how to simulate its physics. */
public class Car {
  /** Constructs a stationary Car at the specified position, and orientation.
   * @param t the position of the Car (velocity is ignored).
   * @param o the orientation of the Car (angular velocity is ignored).
   * @param config a PlayerConfig giving the parameters of the car.
   */
  public Car(Trajectory t, Orientation o, PlayerConfig config) {
    this.config = config;
    this.chassis =
      new RigidBody(300, 300, 75, 300, o.translate(t, 0, 0, 400), o);
    this.wheelFL =
      new RigidBody(50, 4, 4, 4, o.translate(t, -1200, 1500, 500), o);
    this.wheelFR =
      new RigidBody(50, 4, 4, 4, o.translate(t, 1200, 1500, 500), o);
    this.wheelFR.flipXY();
    this.wheelBL =
      new RigidBody(50, 4, 4, 4, o.translate(t, -1200, -1000, 500), o);
    this.wheelBR =
      new RigidBody(50, 4, 4, 4, o.translate(t, 1200, -1000, 500), o);
    this.wheelBR.flipXY();
    this.revs = 50<<8; this.steer = 0;
    this.gear = 1; this.justChangedGear = false;
    this.wx = o.yx << 2; this.wy = o.yy << 2;
  }
  
  /** Constructs a Car at the specified position, stationary, upright and facing north.
   * @param x the x-coordinate of the Car, in units such that the size of the landscape is '1&lt;&lt;32'.
   * @param y the y-coordinate of the Car.
   * @param z the z-coordinate of the Car.
   * @param config a PlayerConfig giving the parameters of the car.
   */
  public Car(int x, int y, int z, PlayerConfig config) {
    this(new Trajectory(x, y, z), Orientation.ID, config);
  }
  
  /* New API. */
  
  /** Returns the Trajectory of the chassis. */
  public Trajectory getTrajectory() { return this.chassis.getTrajectory(); }
  
  /** Advances the physics by 1ms, without collisions. */
  public void tick(int keyMask, boolean withGravity) {
    if (withGravity) {
      this.chassis.accelerate(0, 0, -10<<2);
      this.wheelFL.accelerate(0, 0, -10<<2);
      this.wheelFR.accelerate(0, 0, -10<<2);
      this.wheelBL.accelerate(0, 0, -10<<2);
      this.wheelBR.accelerate(0, 0, -10<<2);
    }
    // Calculate axle vectors for the front wheels.
    final Trajectory t = this.chassis.getTrajectory();
    final Orientation o = this.chassis.getOrientation();
    final int a = Math.max(-128, Math.min(128, (this.steer + (1<<20)) >> 21));
    int ax, ay;
    ax = AX[128+a]; ay = AY[128+a];
    final int axFL = (ax*o.xx + ay*o.yx) >> 14;
    final int ayFL = (ax*o.xy + ay*o.yy) >> 14;
    final int azFL = (ax*o.xz + ay*o.yz) >> 14;
    ax = AX[128-a]; ay = AY[128-a];
    final int axFR = (ax*o.xx - ay*o.yx) >> 14;
    final int ayFR = (ax*o.xy - ay*o.yy) >> 14;
    final int azFR = (ax*o.xz - ay*o.yz) >> 14;
    // Apply forces to keep the wheels on, and work out how fast they're going.
    final int wFL = -this.axle(
      this.wheelFL, o.translate(t, -800, 1500, 40), o, axFL, ayFL, azFL
    );
    final int wFR = this.axle(
      this.wheelFR, o.translate(t, 800, 1500, 40), o, -axFR, -ayFR, -azFR
    );
    final int wBL = -this.axle(
      this.wheelBL, o.translate(t, -800, -1000, 10), o, o.xx, o.xy, o.xz
    );
    final int wBR = this.axle(
      this.wheelBR, o.translate(t, 800, -1000, 10), o, -o.xx, -o.xy, -o.xz
    );
    // Work out whether any drive wheels have fallen off, and if not, how fast
    // the drive shaft is turning.
    boolean dead = false;
    int w = 0, wn = 0;
    if (this.config.fwd) {
      wn += 2;
      if (wFL==(1<<31)) dead = true; else w += wFL;
      if (wFR==(1<<31)) dead = true; else w += wFR;
    }
    if (this.config.bwd) {
      wn += 2;
      if (wBL==(1<<31)) dead = true; else w += wBL;
      if (wBR==(1<<31)) dead = true; else w += wBR;
    }
    if (wn==0) dead = true;
    w = dead ? 0 : w/wn;
    // Update the steering angle.
    final int v = Math.max(w>>4, 1024);
    final int steerQuantum = (1<<29) / v;
    if ((keyMask&(1<<0))!=0 && this.steer>-1<<28) this.steer -= 2*steerQuantum;
    if ((keyMask&(1<<1))!=0 && this.steer<1<<28) this.steer += 2*steerQuantum;
    if (this.steer!=0) this.steer += (this.steer<0 ? 1 : -1)*steerQuantum;
    // Apply engine forces.
    if (!dead && this.revs>(50<<8)) {
      final int revTarget = w * GEARS[this.gear];
      final int et = (Math.max(revTarget, 50<<8) - this.revs) >> 3;
      this.revs += et >> 1;
      final int wt = (-et * GEARS[this.gear]) / wn;
      if (this.config.fwd) {
        this.wheelTorque(wt, axFL, ayFL, azFL, this.wheelFL);
        this.wheelTorque(wt, axFR, ayFR, azFR, this.wheelFR);
      }
      if (this.config.bwd) {
        this.wheelTorque(wt, o.xx, o.xy, o.xz, this.wheelBL);
        this.wheelTorque(wt, o.xx, o.xy, o.xz, this.wheelBR);
      }
    }
    if ((keyMask&(1<<2))!=0) this.revs += ((500<<8) - this.revs) >> 9;
    else this.revs += ((50<<8) - this.revs) >> 10;
    // Apply braking forces if necessary.
    if ((keyMask&(1<<3))!=0) {
      // Do the ABS test.
      int maxW = Integer.MIN_VALUE, minW = Integer.MAX_VALUE;
      if (wFL!=(1<<31)) { if (wFL>maxW) maxW = wFL; if (wFL<minW) minW = wFL; }
      if (wFR!=(1<<31)) { if (wFR>maxW) maxW = wFR; if (wFR<minW) minW = wFR; }
      if (wBL!=(1<<31)) { if (wBL>maxW) maxW = wBL; if (wBL<minW) minW = wBL; }
      if (wBR!=(1<<31)) { if (wBR>maxW) maxW = wBR; if (wBR<minW) minW = wBR; }
      if (minW>0) minW = maxW - (maxW>>4) - ABS_THRESHOLD;
      if (maxW<0) maxW = minW - (minW>>4) + ABS_THRESHOLD;
      if (wFL>=minW && wFL<=maxW && wFR>=minW && wFR<=maxW) {
        this.wheelTorque(wFL>0?-1500:1500, axFL, ayFL, azFL, this.wheelFL);
        this.wheelTorque(wFR>0?-1500:1500, axFR, ayFR, azFR, this.wheelFR);
      }
      if (wBL>=minW && wBL<=maxW && wBR>=minW && wBR<=maxW) {
        this.wheelTorque(wBL>0?-500:500, o.xx, o.xy, o.xz, this.wheelBL);
        this.wheelTorque(wBR>0?-500:500, o.xx, o.xy, o.xz, this.wheelBR);
      }
    }
    // Change gear if necessary.
    if ((keyMask&(1<<4))!=0) {
      if (!this.justChangedGear) {
        int newGear = 0, best = Integer.MAX_VALUE;
        for (int i=0; i<GEARS.length; i++) if (i!=this.gear) {
          int error = Math.abs((250<<8)/GEARS[i] - w);
          if (error<best) { best = error; newGear = i; }
        }
        this.gear = newGear;
        System.out.println(
	  this.config.name+": "+
	  (this.gear==0 ? "gear = R" : "gear = "+this.gear)
	);
      }
      this.justChangedGear = true;
    } else {
      this.justChangedGear = false;
    }
    // Apply aerodynamic forces.
    this.aerofoil(
      o.translate(t, 0, -1250, 750),
      (8*o.zx-o.yx) >> 10, (8*o.zy-o.yy) >> 10, (8*o.zz-o.yz) >> 10
    );
    // Move everything.
    this.chassis.tick();
    this.wheelFL.tick();
    this.wheelFR.tick();
    this.wheelBL.tick();
    this.wheelBR.tick();
    // Update Camera angle.
    this.wx += t.vx >> 6; this.wy += t.vy >> 6;
    final int f = (this.wx*this.wx + this.wy*this.wy) >> 16;
    this.wx -= (this.wx*f) >> 17; this.wy -= (this.wy*f) >> 17;
  }
  
  /** Describes this Car using 10 CollisionSpheres, one for each wheel and six for the chassis.
   * @param spheres an array in which to place the spheres.
   * @param used the index into the array at which to place the first sphere.
   * @return the index after the last sphere.
   */
  public int getSpheres(CollisionSphere[] spheres, int used) {
    Trajectory t = this.chassis.getTrajectory();
    Orientation o = this.chassis.getOrientation();
    spheres[used++] = new CollisionSphere(
      o.translate(t,    0, 2000,    0), o, 250, 1, 10, 1<<7, 20, this.chassis
    );
    spheres[used++] = new CollisionSphere(
      o.translate(t,    0, 1000,  125), o, 375, 1, 10, 1<<7, 20, this.chassis
    );
    spheres[used++] = new CollisionSphere(
      o.translate(t,    0,    0,  500), o, 250, 1, 10, 1<<7, 20, this.chassis
    );
    spheres[used++] = new CollisionSphere(
      o.translate(t,    0, -875,  125), o, 375, 1, 10, 1<<7, 20, this.chassis
    );
    spheres[used++] = new CollisionSphere(
      o.translate(t, -750,    0,    0), o, 250, 1, 10, 1<<7, 20, this.chassis
    );
    spheres[used++] = new CollisionSphere(
      o.translate(t,  750,    0,    0), o, 250, 1, 10, 1<<7, 20, this.chassis
    );
    t = this.wheelFL.getTrajectory();
    o = this.wheelFL.getOrientation();
    spheres[used++] = new CollisionSphere(
      o.translate(t, 100, 0, 0), o, 500, 1, 2, WCOF, 2, this.wheelFL
    );
    t = this.wheelFR.getTrajectory();
    o = this.wheelFR.getOrientation();
    spheres[used++] = new CollisionSphere(
      o.translate(t, 100, 0, 0), o, 500, 1, 2, WCOF, 2, this.wheelFR
    );
    t = this.wheelBL.getTrajectory();
    o = this.wheelBL.getOrientation();
    spheres[used++] = new CollisionSphere(
      o.translate(t, 100, 0, 0), o, 500, 1, 2, WCOF, 2, this.wheelBL
    );
    t = this.wheelBR.getTrajectory();
    o = this.wheelBR.getOrientation();
    spheres[used++] = new CollisionSphere(
      o.translate(t, 100, 0, 0), o, 500, 1, 2, WCOF, 4, this.wheelBR
    );
    return used;
  }
  
  /** Renders this Car using the 'Model.WHEEL' and 'Model.CAR[team]' models, and adds the resulting queue of Faces to 'f'.
   * @param inCar if 'true', 'Models.COCKPIT' is substituted for 'Models.CAR'.
   * @param f the Frame to which to add the Faces.
   */
  public void addFacesTo(final Frame f, final boolean inCar) {
    Model m = (inCar ? Model.COCKPITS : Model.CARS)[this.config.colour];
    m.project(this.chassis.getTrajectory(), this.chassis.getOrientation(), f);
    m = Model.WHEEL;
    m.project(this.wheelFL.getTrajectory(), this.wheelFL.getOrientation(), f);
    m.project(this.wheelFR.getTrajectory(), this.wheelFR.getOrientation(), f);
    m.project(this.wheelBL.getTrajectory(), this.wheelBL.getOrientation(), f);
    m.project(this.wheelBR.getTrajectory(), this.wheelBR.getOrientation(), f);
  }
  
  /** Returns the engine speed in revolutions per minute. */
  public int getRevs() {
    return this.revs / (int)((1<<8)*2*Math.PI/60);
  }
  
  /** Returns the currently selected gear, with '0' meaning reverse. */
  public int getGear() { return this.gear; }
  
  /** Returns a Camera describing a viewpoint which follows a few metres behind this Car. The Camera generally looks in the direction in which the Car is moving, and always remains upright. The Camera is raised if necessary to avoid the Landscape.
   * @param dist the distance of the Camera from the Car, in units such that the size of the landscape is '1&lt;&lt;32'.
   * @param land a Landscape that the Camera should not dip below.
   */
  public Camera getCamera(int dist, Landscape land) {
    final Orientation o = this.chassis.getOrientation();
    final Trajectory t = o.translate(this.chassis.getTrajectory(), 0, 500, 500);
    final int cx = t.x - this.wx * (dist>>16);
    final int cy = t.y - this.wy * (dist>>16);
    final int cz = Math.max(land.getHeight(cx, cy) + (1<<20), t.z + (dist>>2));
    return new Camera(cx, cy, cz, this.wx, this.wy);
  }
  
  /** Returns a Camera describing the driver's viewpoint. Apart from remaining level, the Camera looks along the y-axis of 'chassis', and rotates following the z-axis of 'chassis'. */
  public Camera getFixedCamera() {
    final Orientation o = this.chassis.getOrientation();
    final Trajectory t = o.translate(this.chassis.getTrajectory(), 0, 500, 500);
    // Work out a normal vector in the direction 'o.yx, o.yy'.
    int wx = o.yx, wy = o.yy;
    int f = wx; if (f<-wx) f = -wx; if (f<wy) f = wy; if (f<-wy) f = -wy;
    if (f==0) f = 1; // This is a hack and Oggie is ashamed.
    wx = (wx<<15) / f; wy = (wy<<15) / f;
    for (int i=0; i<5; i++) {
      f = (wx*wx + wy*wy - (1<<30)) >> 16;
      wx -= (f*wx) >> 15; wy -= (f*wy) >> 15;
    }
    // Work out another normal vector.
    int sin = (o.zx*wy - o.zy*wx) >> 15, cos = o.zz;
    f = sin; if (f<-sin) f = -sin; if (f<cos) f = cos; if (f<-cos) f = -cos;
    if (f==0) f = 1; // This is a hack and Oggie is ashamed.
    sin = (sin<<15) / f; cos = (cos<<15) / f;
    for (int i=0; i<5; i++) {
      f = (sin*sin + cos*cos - (1<<30)) >> 16;
      sin -= (f*sin) >> 15; cos -= (f*cos) >> 15;
    }
    return new Camera(t.x, t.y, t.z, wx<<1, wy<<1, sin<<1, cos<<1);
  }
  
  /* Private. */
  
  /** Applies equal and opposite forces to 'this.chassis' and 'wheel' so as to attach the wheel to the chassis at the specified point.
   * @param wheel the RigidBody to attach.
   * @param t the Trajectory describing the point of attachment.
   * @param o the Orientation describing the point of attachment.
   * @param ax the x-component, in the world's frame, of a vector of length '1&lt;&lt;14' along the desired wheel x-axis. This may differ from the x-axis of 'o' due to steering.
   * @param ay the y-component of the axle vector.
   * @param az the z-component of the axle vector.
   * @return the component of the wheel's angular velocity (relative to the chassis) in the direction of the axle; that is, how fast the axle is spinning. The units are such that one radian per second is '1&lt;&lt;8'. Returns '1&lt;&lt;31' if the wheel has fallen off.
   */
  private int axle(
    RigidBody wheel,
    Trajectory t, Orientation o,
    int ax, int ay, int az
  ) {
    final Orientation wo = wheel.getOrientation();
    final Trajectory wt = wo.translate(wheel.getTrajectory(), 400, 0, 0);
    // Separate the components of the displacement (wheel relative to chassis)
    // normal and tangential to the chassis' xy-plane, in millimetres.
    final int x = (wt.x - t.x) >> 12;
    final int y = (wt.y - t.y) >> 12;
    final int z = (wt.z - t.z) >> 12;
    final int B = 500;
    if (x>B || x<-B || y>B || y<-B || z>B || z<-B) return 1<<31;
    final int n = (x*o.zx + y*o.zy + z*o.zz) >> 14;
    final int xt = x - ((n*o.zx) >> 14);
    final int yt = y - ((n*o.zy) >> 14);
    final int zt = z - ((n*o.zz) >> 14);
    // Separate the components of the relative velocity, in mm/s.
    final int vx = (wt.vx - t.vx) >> 2;
    final int vy = (wt.vy - t.vy) >> 2;
    final int vz = (wt.vz - t.vz) >> 2;
    final int vn = (vx*o.zx + vy*o.zy + vz*o.zz) >> 14;
    final int vxt = vx - ((vn*o.zx) >> 14);
    final int vyt = vy - ((vn*o.zy) >> 14);
    final int vzt = vz - ((vn*o.zz) >> 14);
    // Apply forces to correct the displacement.
    final int fn = (n + ((100*vn)>>10)) * (10<<2);
    final int fx = (xt + ((20*vxt)>>10)) * (100<<2) + ((fn * o.zx) >> 14);
    final int fy = (yt + ((20*vyt)>>10)) * (100<<2) + ((fn * o.zy) >> 14);
    final int fz = (zt + ((20*vzt)>>10)) * (100<<2) + ((fn * o.zz) >> 14);
    this.chassis.applyForce(fx, fy, fz, wt.x, wt.y, wt.z);
    wheel.applyForce(-fx, -fy, -fz, wt.x, wt.y, wt.z);
    // Work out the angle between the wheel's x-axis and the axle.
    final int wax = (ay*wo.xz - az*wo.xy) >> 14;
    final int way = (az*wo.xx - ax*wo.xz) >> 14;
    final int waz = (ax*wo.xy - ay*wo.xx) >> 14;
    // Separate the components of the angular velocity (wheel relative to
    // chassis) parallel and perpendicular to the axle.
    final int wx = wo.wx - o.wx;
    final int wy = wo.wy - o.wy;
    final int wz = wo.wz - o.wz;
    final int wn = (wx*ax + wy*ay + wz*az) >> 14;
    final int wxt = wx - ((wn*ax) >> 14);
    final int wyt = wy - ((wn*ay) >> 14);
    final int wzt = wz - ((wn*az) >> 14);
    // Apply torques to correct the angle.
    final int tx = ((wax + 5*wxt) * 20000) >> 14;
    final int ty = ((way + 5*wyt) * 20000) >> 14;
    final int tz = ((waz + 5*wzt) * 20000) >> 14;
    this.chassis.applyTorque(tx, ty, tz);
    wheel.applyTorque(-tx, -ty, -tz);
    return wn;
  }
  
  /** Applies a torque with the specified magnitude and direction to the chassis, and applies an equal and opposite torque to the specified wheel. */
  private void wheelTorque(int t, int ax, int ay, int az, RigidBody wheel) {
    final int tx = (t*ax) >> 14, ty = (t*ay) >> 14, tz = (t*az) >> 14;
    this.chassis.applyTorque(tx, ty, tz);
    wheel.applyTorque(-tx, -ty, -tz);
  }
  
  /** Applies a force to 'this.chassis' to represent the aerodynamic force on an aerofoil. The orientation and area of the aerofoil are represented by the direction and length of a normal vector. A length of '1&lt;&lt;8' represents an area of 1m2.
   * @param at the Trajectory of the centre of the aerofoil.
   * @param nx the x-component of the normal vector.
   * @param ny the y-component of the normal vector.
   * @param nz the z-component of the normal vector.
   */
  private void aerofoil(Trajectory at, int nx, int ny, int nz) {
    // Work out air flow. '1<<4' is 1m3/s.
    final int nv = (at.vx*nx + at.vy*ny + at.vz*nz) >> 16;
    // Work out normal velocity squared. '1' is (1m/s)2.
    final int nv2 = nv * (nv<0?nv:-nv) / ((nx*nx + ny*ny + nz*nz) >> 8);
    // Work out force. '1<<2' is 1N.
    final int fx = (nv2*nx) >> 6, fy = (nv2*ny) >> 6, fz = (nv2*nz) >> 6;
    this.chassis.applyForce(fx, fy, fz, at.x, at.y, at.z);
  }
  
  /** A look-up table for steering. */
  private static final int[] AX, AY;
  static{
    AX = new int[257]; AY = new int[257];
    AX[128] = 1<<14; AY[128] = 0;
    for (int i=0; i<257; i++) if (i!=128) {
      double x = 2500 / Math.tan(0.005 * (i-128));
      double a = Math.atan(2500 / (800 + x));
      AX[i] = (int)Math.round((1<<14) * Math.cos(a));
      AY[i] = (int)Math.round((1<<14) * -Math.sin(a));
    }
  }
  
  /** The gearbox ratios. */
  private static final int[] GEARS = new int[] { -10, 10, 7, 5, 4, 3};
  
  /** The wheel coefficient of friction times '1&lt;&lt;8'. */
  private static final int WCOF = 0x180;
  
  /** The amount by which the maximum and minimum angular velocities of the wheels may differ without triggering the ABS. This amount is increased by half the angular velocity of the fastest wheel before applying the test. The units are such that one radian per second is '1&lt;&lt;8'. */
  private static final int ABS_THRESHOLD = 0x100;
  
  private PlayerConfig config;

  private RigidBody chassis;
  private RigidBody wheelFL;
  private RigidBody wheelFR;
  private RigidBody wheelBL;
  private RigidBody wheelBR;
  
  /** The engine speed, in units such that one radian per second is '1&lt;&lt;8'. */
  private int revs;
  
  /** The steering angle, from '-1&lt;&lt;28' to '1&lt;&lt;28'. The angle is proportional to the square of this value (but of the same sign as this value). */
  private int steer;
  
  /** The current gear. */
  private int gear;
  
  /** 'true' if 'change gear' was pressed last frame. */
  private boolean justChangedGear;
  
  /** A component of the vector of length '1&lt;&lt;16' in the direction the camera returned by 'getCamera()' is looking. */
  private int wx, wy;
  
  /* Test code. */
  
  public static void main(String[] args) {
    if (args.length<1 || args.length>2) throw new IllegalArgumentException(
      "Syntax: java org.sc3d.apt.jrider.v1.Car <seed> [<slow>]"
    );
    final int slow = args.length>=2 ? Integer.parseInt(args[1]) : 1;
    final Landscape land = Landscape.generate(args[0], 11);
    final int numCars = 6;
    final Lens lens = new Lens(256, 256, 128, 256);
    final SceneImage si = new SceneImage(lens, 512, 256, "Testing Car");
    final Car[] me = new Car[numCars];
    for (int i=0; i<numCars; i++) {
      final PlayerConfig config = new PlayerConfig(
        false, false,
	i, "Car "+i
      );
      me[i] = new Car(0, 0, (1<<29)+(i<<24), config);
    }
    final CollisionSphere[] CSS = new CollisionSphere[numCars*10];
    long time = System.currentTimeMillis()+1000;
    final FPSCounter fpsc = new FPSCounter();
    boolean quit = false;
    while (!quit) {
      // Choose a camera position.
      final Trajectory t = me[0].chassis.getTrajectory();
      final int wx = (3<<16)/5, wy = (-4<<16)/5;
      final int cx = t.x - (wx<<10), cy = t.y - (wy<<10);
      final int cz = Math.max(t.z + (1<<24), land.getHeight(cx, cy) + (1<<20));
      final Camera c = new Camera(cx, cy, cz, wx, wy);
      // Draw a frame.
      final Frame f = si.reset(c, land);
      for (int i=0; i<numCars; i++) me[i].addFacesTo(f, false);
      si.doFrame();
      // Think repeatedly to catch up with real time.
      long nextTime = System.currentTimeMillis();
      fpsc.doFrame((int)(nextTime-time), 1);
      while (time<nextTime) {
        int used = 0;
        for (int i=0; i<numCars; i++) used = me[i].getSpheres(CSS, used);
        for (int i=0; i<used; i++) {
          CSS[i].hitLand(land);
          for (int j=0; j<i; j++) CSS[i].hitSphere(CSS[j]);
        }
        for (int i=0; i<numCars; i++) me[i].tick(0, true);
        time += slow;
      }
    }
  }
}
