package org.sc3d.apt.jrider.v1;

import java.awt.image.IndexColorModel;

/** This class provides tools for generating landscapes and modifying them. */
public class LandGen {
  /** Constructs a LandGen that generates a landscape of the specified size.
   * @param rand a source of random numbers. If two LandGens are constructed with the same seed and size and the same operations are performed on them, then the results will be the same.
   * @param log2size the log to base 2 of the size of the landscape in squares, which should be at least 3 (i.e. the landscape should be at least 8x8).
   */
  public LandGen(Random rand, int log2size) {
    this.log2size = log2size; this.size = 1<<log2size;
    // Generate a fractal height-field.
    int roughness = rand.nextUInt(1<<29);
    int height = rand.nextUInt(1<<29);
    int size = 4;
    int[] h = new int[size*size];
    for (int i=0; i<size*size; i++) h[i] = rand.nextSInt(height);
    while (size<this.size) {
      int size2 = size*2;
      int[] h2 = new int[size2*size2];
      roughness = rand.nextUInt(roughness>>1) + (roughness>>2);
      height += roughness;
      for (int y0=0; y0<size; y0++) for (int x0=0; x0<size; x0++) {
        int x1 = (x0+1) & (size-1), y1 = (y0+1) & (size-1);
        h2[x0*2+size2*y0*2] =
          (2*h[x0+size*y0] + h[x1+size*y0] + h[x0+size*y1] + 2)/4 +
          rand.nextSInt(roughness);
        h2[x0*2+1+size2*y0*2] =
          (2*h[x1+size*y0] + h[x0+size*y0] + h[x1+size*y1] + 2)/4 +
          rand.nextSInt(roughness);
        h2[x0*2+size2*(y0*2+1)] =
          (2*h[x0+size*y1] + h[x1+size*y1] + h[x0+size*y0] + 2)/4 +
          rand.nextSInt(roughness);
        h2[x0*2+1+size2*(y0*2+1)] =
          (2*h[x1+size*y1] + h[x0+size*y1] + h[x1+size*y0] + 2)/4 +
          rand.nextSInt(roughness);
      }
      size = size2; h = h2;
    }
    this.h = h;
    // Generate appropriate colours;
    this.c = new byte[size*size];
    final int CDH = height / 0x30;
    final int M = size-1;
    for (int y=0; y<size; y++) for (int x=0; x<size; x++) {
      int c = 0x48 + (
        h[x+size*y] +
        h[((7*x+2*y)&M)+size*((7*y-2*x)&M)]
      ) / CDH;
      this.c[x+size*y] = (byte)(c>0x5f ? 0x5f : c<0x30 ? 0x30 : c);
    }
  }
  
  /* New API. */
  
  /** Returns the size of the landscape in squares. */
  public int getSize() { return this.size; }
  
  /** Returns the height of the specified point.
   * @param x the x-coordinate (east) of the point in units such that the size of the landscape is '1&lt;&lt;32'.
   * @param y the y-coordinate (north) of the point.
   * @return the height.
   */
  public int getHeight(int x, int y) {
    return this.h[
      (x>>>(32-this.log2size)) +
      this.size*(y>>>(32-this.log2size))
    ];
  }
  
  /** Returns the colour of the specified point.
   * @param x the x-coordinate (east) of the point in units such that the size of the landscape is '1&lt;&lt;32'.
   * @param y the y-coordinate (north) of the point.
   * @return the colour.
   */
  public byte getColour(int x, int y) {
    return this.c[
      (x>>>(32-this.log2size)) +
      this.size*(y>>>(32-this.log2size))
    ];
  }
  
  /** Returns a reference to the height field. */
  public int[] getHeights() { return this.h; }
  
  /** Returns a reference to the colour field. */
  public byte[] getColours() { return this.c; }
  
  /** The mapping from byte values to colours. */
  public static final IndexColorModel ICM;
  static {
    byte[] r = new byte[0x100], g = new byte[0x100], b = new byte[0x100];
    // Greys.
    for (int i=0; i<6; i++) {
      r[i*8+0x07] = g[i*8+0x07] = b[i*8+0x07] = (byte)(i*0x33); }
    // Grass colours.
    for (int i=0; i<6; i++) {
      r[i*8+0x37] = (byte)(i*0x11);
      g[i*8+0x37] = (byte)(0x80-i*0x11);
      b[i*8+0x37] = (byte)0x00;
    }
    // Tree colours.
    r[0x67] = (byte)0x20; g[0x67] = (byte)0x80; b[0x67] = (byte)0x20;
    r[0x6f] = (byte)0x40; g[0x6f] = (byte)0xc0; b[0x6f] = (byte)0x80;
    r[0x77] = (byte)0x00; g[0x77] = (byte)0x60; b[0x77] = (byte)0x20;
    r[0x7f] = (byte)0x80; g[0x7f] = (byte)0x80; b[0x7f] = (byte)0x00;
    // Brown for cockpit.
    r[0x87] = (byte)0x80; g[0x87] = (byte)0x40; b[0x87] = (byte)0x00;
    // Car colours.
    r[0x8f] = (byte)0xff; g[0x8f] = (byte)0x00; b[0x8f] = (byte)0x00;
    r[0x97] = (byte)0x00; g[0x97] = (byte)0xff; b[0x97] = (byte)0x00;
    r[0x9f] = (byte)0xff; g[0x9f] = (byte)0xff; b[0x9f] = (byte)0x00;
    r[0xa7] = (byte)0x00; g[0xa7] = (byte)0x00; b[0xa7] = (byte)0xff;
    r[0xaf] = (byte)0xff; g[0xaf] = (byte)0x00; b[0xaf] = (byte)0xff;
    r[0xb7] = (byte)0x00; g[0xb7] = (byte)0xff; b[0xb7] = (byte)0xff;
    // Steel colour for water.
    r[0xbf] = (byte)0x80; g[0xbf] = (byte)0xa0; b[0xbf] = (byte)0xc0;
    // Make 8 shades of all of the above.
    for (int i=0; i<0xe0; i+=8) for (int j=0; j<7; j++) {
      r[i+j] = (byte)(((r[i+7]&0xff)*(j+2)+4)/9);
      g[i+j] = (byte)(((g[i+7]&0xff)*(j+2)+4)/9);
      b[i+j] = (byte)(((b[i+7]&0xff)*(j+2)+4)/9);
    }
    // Fire colours.
    for (int i=0; i<16; i++) {
      r[i+0xc0] = (byte)0xff;
      g[i+0xc0] = (byte)(i*0x11);
      b[i+0xc0] = (byte)0x00;
      r[i+0xd0] = (byte)0xff;
      g[i+0xd0] = (byte)0xff;
      b[i+0xd0] = (byte)(i*0x11);
    }
    // Sky colours.
    for (int i=0; i<32; i++) {
      r[i+0xe0] = (byte)(0x7c-i*4);
      g[i+0xe0] = (byte)(0xcc-i*4);
      b[i+0xe0] = (byte)(0xfc-i*4);
    }
    // Make the IndexColorModel.
    ICM = new IndexColorModel(8, 0x100, r, g, b);
  }
  
  /** Smooths the landscape by averaging 3x3 squares. */
  public void smooth() {
    final int S2 = this.size * this.size, M = S2-1;
    int[] h = this.h; this.h = new int[S2];
    for (int i=0; i<S2; i++) {
      this.h[i] = (
        h[(i-1-this.size)&M] +
        h[(i-this.size)&M] +
        h[(i+1-this.size)&M] +
        h[(i-1)&M] +
        h[(i+1)&M] +
        h[(i-1+this.size)&M] +
        h[(i+this.size)&M] +
        h[(i+1+this.size)&M] +
        4 - 8*h[i]
      )/9 + h[i];
    }
  }
  
  /** Draws a parabolic lump that looks like a tree.
   * @param x the x-coordinate (east) of the lump, in units such that the size of the landscape is '1&lt;&lt;32'.
   * @param y the y-coordinate (north) of the lump.
   * @param z the z-coordinate (up) of the lump.
   * @param r the radius of the lump.
   * @param h the height of the lump.
   * @param colour a colour number interpreted according to 'ICM'.
   */
  public void drawTree(
    final int x, final int y, final int z,
    final int r, final int h,
    final byte colour
  ) {
    final int shift = 32-this.log2size;
    final int half = 1 << (shift-1);
    final int rsq = r >> (shift-1);
    if (r<(1<<16)) return; // Hack.
    final int ratio = h / ((r>>16) * (r>>16));
    for (int dysq=-rsq; dysq<=rsq; dysq++) {
      for (int dxsq=-rsq; dxsq<=rsq; dxsq++) {
        final int xsq = (x + (dxsq<<shift)) >>> shift;
        final int ysq = (y + (dysq<<shift)) >>> shift;
        final int dx = (x - (xsq<<shift) - half) >> 16;
        final int dy = (y - (ysq<<shift) - half) >> 16;
        final int dz = h - (dx*dx + dy*dy) * ratio;
        final int index = xsq + this.size*ysq;
        if (z+dz>this.h[index]) {
          this.h[index] = z + dz; this.c[index] = colour;
        }
      }
    }
  }
  
  /** Draws vegetation by calling 'drawTree()' repeatedly. This method will try to draw trees in sensible places (i.e. not on top of a road).
   * @param rand a source of random numbers.
   */
  public void addTrees(final Random rand) {
    // Choose three fertility thresholds.
    int[] f = new int[3];
    for (int i=0; i<3; i++) {
      int newF = rand.nextUInt(0x30) + rand.nextUInt(0x30);
      int j = i; while (j>0 && f[j-1]<newF) { f[j] = f[j-1]; j--; }
      f[j] = newF;
    }
    // Loop for each clump of trees.
    final int numClumps = rand.nextUInt(200);
    for (int i=0; i<numClumps; i++) {
      int x = rand.nextInt(), y = rand.nextInt();
      final int numTrees = rand.nextUInt(50);
      final int fertility =
        0x60 - (0xff & this.getColour(x, y)) + rand.nextUInt(0x30);
      // Loop for each tree.
      for (int j=0; j<numTrees; j++) {
        x += rand.nextSInt(1<<27); y += rand.nextSInt(1<<27);
        final byte colour = this.getColour(x, y);
        if (colour>=0x30 && colour<0x60) {
          final int z = this.getHeight(x, y);
          final int age = rand.nextUInt(1<<22);
          if (fertility>f[0]) { // Oak.
            this.drawTree(x, y, z, age<<3, age<<4, (byte)0x60);
          } else if (fertility>f[1]) { // Poplar.
            this.drawTree(x, y, z, age<<2, age<<4, (byte)0x68);
          } else if (fertility>f[2]) { // Pine.
            this.drawTree(x, y, z, age<<2, age<<2, (byte)0x70);
            this.drawTree(x, y, z, age<<1, age<<3, (byte)0x70);
          } else { // Heather.
            for (int k=0; k<5; k++) {
              this.drawTree(
                x+rand.nextSInt(1<<24), y+rand.nextSInt(1<<24), z,
                age<<1, age, (byte)0x78
              );
            }
          }
        }
      }
    }
  }
  
  /** Draws a loop of road on the landscape.
   * @param cs an array of Cambers describing the path of the road.
   */
  public void drawRoad(final Camber[] cs) {
    // Loop through the Cambers.
    Camber c2 = cs[cs.length-1];
    for (int i=0; i<cs.length; i++) {
      Camber c1 = c2; c2 = cs[i];
      // Draw the tarmac.
      this.drawQuad(c1, c2, -192, 0, 192, 256, 0, (byte)0x10);
      // Draw the left barrier or kerb.
      if (c1.barrierLeft || c2.barrierLeft) {
        this.drawQuad(c1, c2, -256, 0, -192, 128, 1<<23, (byte)0x88);
        this.drawQuad(c1, c2, -256, 128, -192, 256, 1<<23, (byte)0x20);
      } else {
        this.drawQuad(c1, c2, -256, 0, -192, 256, 0, (byte)0x18);
      }
      // Draw the right barrier or kerb.
      if (c1.barrierRight || c2.barrierRight) {
        this.drawQuad(c1, c2, 192, 0, 256, 128, 1<<23, (byte)0x20);
        this.drawQuad(c1, c2, 192, 128, 256, 256, 1<<23, (byte)0x88);
      } else {
        this.drawQuad(c1, c2, 192, 0, 256, 256, 0, (byte)0x18);
      }
    }
    // Draw the starting grid.
    this.drawQuad(cs[0], cs[1], -192, 0, 192, 64, 0, (byte)0x28);
    c2 = cs[0];
    for (int i=cs.length-1; i>=cs.length-3; i--) {
      Camber c1 = cs[i];
      for (int j=-96; j<=96; j+=192) {
        this.drawQuad(c1, c2, j-64, 0, j-48, 48, 0, (byte)0x28);
        this.drawQuad(c1, c2, j+48, 0, j+64, 48, 0, (byte)0x28);
        this.drawQuad(c1, c2, j-64, 48, j+64, 64, 0, (byte)0x28);
      }
      c2 = c1;
    }
  }
  
  /** Floods the landscape to a depth of 'waterLevel'. Squares that are underwater are coloured with one of the colours from '0xB8' to '0xBF' depending on the depth. */
  public void addWater(int waterLevel) {
    int seed = 1;
    for (int i=0; i<this.h.length; i++) {
      final int depth = waterLevel - this.h[i];
      if (depth>0) {
        seed *= 836543875;
        final int band = depth+(seed>>>(32-22)) >> 22;
        this.c[i] = (byte)(band<8 ? 0xBF-band : 0xB8);
        this.h[i] = waterLevel;
      }
    }
  }
  
  /* Private. */
  
  /** Draws a quadralateral. The height is bilinearly interpolated, and the colour is constant. This method is primarily designed for drawing bits of road.
   * <p>A coordinate system is specified using two Cambers. '-256, 0' and '256, 0' mean the left- and right-hand sides of 'c1', and '-256, 256' and '256, 256' mean the left- and right-hand sides of 'c2'. The corners of the quadralateral are then specified using this coordinate system.
   * <p>The quadralateral is plotted simply by plotting a grid of points. The grid spacing in each direction is fixed, and has been chosen so that at least one point lands in each square of the landscape, given Cambers 16m apart describing a road 32m wide.
   * @param c1 the first Camber.
   * @param c2 the second Camber.
   * @param l the left-hand side of the quadralateral.
   * @param t the top of the quadralateral (nearest 'c1').
   * @param r the right-hand side of the quadralateral.
   * @param b the bottom of the quadralateral (nearest 'c2').
   * @param rise the height of the quadralateral above the road, in units such that the size of the landscape is '1&lt;&lt;32'.
   * @param colour the colour of the quadralateral.
   */
  private void drawQuad(
    final Camber c1, final Camber c2,
    final int l, final int t, final int r, final int b,
    final int rise,
    final byte colour
  ) {
    final int shift = 32-this.log2size;
    final int dx = (c2.x-c1.x)>>8, dy = (c2.y-c1.y)>>8, dz = (c2.z-c1.z)>>8;
    final int dxx=(c2.xx-c1.xx)>>8, dxy=(c2.xy-c1.xy)>>8, dxz=(c2.xz-c1.xz)>>8;
    // These are fudged for drawing 16m of road 32m wide.
    final int stepi = 1 << (8-(this.log2size-6+1));
    final int stepj = 1 << (8-(this.log2size-6+1));
    // Loop through the grid of points.
    for (int j=t; j<=b; j+=stepj) {
      final int xj = c1.x + j*dx, yj = c1.y + j*dy, zj = c1.z + j*dz;
      final int xxj = c1.xx + j*dxx, xyj = c1.xy + j*dxy, xzj = c1.xz + j*dxz;
      for (int i=l; i<=r; i+=stepi) {
        final int x = xj + i*(xxj>>8), y = yj + i*(xyj>>8), z = zj + i*(xzj>>8);
        final int index = (x>>>shift) + this.size*(y>>>shift);
        this.h[index] = z + rise; this.c[index] = colour;
      }
    }
  }
  
  private int log2size;
  private int size;
  private int[] h;
  private byte[] c;
  
  /* Test code. */
  
  public static void main(String[] args) {
    if (args.length!=1) throw new IllegalArgumentException(
      "Syntax: java org.sc3d.apt.jrider.v1.LandGen <seed>"
    );
    LandGen me = new LandGen(new Random(args[0]), 8);
    me.smooth(); me.smooth();
    // Display the result.
    org.sc3d.apt.screen.v4.Screen screen =
      new org.sc3d.apt.screen.v4.Screen(256, 256, "Testing LandGen");
    int[] h = me.getHeights();
    int[] pixels = screen.getPixels();
    for (int i=0; i<256*256; i++) pixels[i] = 0x010101 * (h[i]/(1<<20)+128);
    screen.doFrame();
  }
}
