package org.sc3d.apt.jrider.v1;

import java.awt.event.*;
import java.io.*;

/** A subclass of Controller for playing over a network. Keypresses are sent asynchronously to the server, and a properly synchronized stream of data comes back, which is then expanded to one mask per millisecond.
 * <p>The protocol used to talk to the server is described properly in a separate document 'protocol.h'. The protocol starts with a sequence that configures the number of players, the colours of their cars, the name of the track, and so on. It then settles down to a simpler protocol for sending keypresses back and forth, which continues until the end of the game, at which point the connection is closed. However, some configuration messages need to be sent in the middle of the game too, for example when a new player joins.
 * <p>This class itself only understands the simple part of the protocol concerning keypresses. It passes server-to-client extended messages to a NetworkListener in a fairly raw form, and it provides a method for sending client-to-server extended messages, again in a fairly raw form. The caller should use these hooks to handle the logically complicated parts of the protocol.
 */
public class NetworkController extends Controller {
  /** Constructs a NetworkController.
   * @param keyCodes an array of length 31 giving the virtual key code for each of the logical keys. Any entry may be '-1' to indicate that no key is defined.
   * @param in the InputStream from the game server.
   * @param out the OutputStream to the game server. You should ensure there are no transmission delays (i.e. call 'Socket.setTcpNoelay(true)' or equivalent).
   */
  public NetworkController(
    int[] keyCodes,
    InputStream in, OutputStream out
  ) {
    super(keyCodes);
    this.in = in; this.out = out;
    this.listener = null;
    this.msPerWord = 10;
    this.pNums = new int[6];
    for (int i=0; i<this.pNums.length; i++) this.pNums[i] = -1;
  }

  /* New API. */
  
  /** Sends a client-to-server extended message. The message will be inserted into the keypress data in a thread-safe way.
   * <p>The possible message types are:<ul>
   * <li>type 0 (cs_chat_message), a chat message. The payload is an UTF-8 string, and the server will forward it to other players.
   * <li>type 1 (cs_list_tracks), request that the server list the tracks on which games are currently being played. No payload.
   * <li>type 2 (cs_choose_track), choose track. The payload is an UTF-8 string giving the name of the track. The name need not be canonicalised. When you send a message, the server behaves as if you also send a message of type 3 (cs_claim_player).
   * <li>type 3 (cs_claim_player), request that the server allocate us a player number. No payload.
   * <li>type 4 (cs_choose_car_parameters), change a player's configuration. Also, use this to add a player to a game for the first time. The payload consists of at least two bytes. The first is the player number, from '0' to '5'. The second byte is interpreted as follows: the top bit is set for front- and four-wheel drive; the next bit is set for back- and four-wheel drive; the bottom six bits give the car colour from '0' to '5'. The remaining bytes form a UTF-8 string giving the player's name.
   * <li>type 5 (cs_send_checksums), send a checksum of the track. This is a string of four bytes calculated from 'Version.CHECKSUM' (which is a hash of the source code for the jrider package) and 'Landscape.getCheckSum()'.
   * </ul>Most messages can be sent at any time. However, 'choose track' should be sent just once, and before claiming any players, and before sending the track checksum. Also, you should only send 'configure car' messages (and key data for keys other than 'escape') for player numbers that the server has allocated to you.
   * <p>This method has a rather abrupt behaviour in the face of an IOException: it calls 'System.exit(0)'. This should probably be fixed.
   * @param type the extended message type.
   * @param payload the message payload, as an array of bytes. The length of the array is interpreted as the length of the payload. 'null' may be passed and will be treated as an array of length '0'.
   */
  public synchronized void sendExtendedMessage(int type, byte[] payload) {
    System.out.println("sendExtendedMessage(");
    System.out.println("  type="+type);
    if (payload==null) {
      System.out.println("  payload=null");
    } else {
      System.out.print("  payload={");
      if (payload.length>0) {
        System.out.print(Integer.toHexString(0xFF&payload[0]));
        for (int i=1; i<payload.length; i++) {
          System.out.print(" "+Integer.toHexString(0xFF&payload[i]));
        }
      }
      System.out.println("}");
    }
    System.out.println(")");
    final int length = payload==null ? 0 : payload.length;
    if (type<0 || type>=0x40 || length>0xFFFF)
      throw new IllegalArgumentException("Bad client-to-server message");
    final byte[] message = new byte[length+3];
    message[0] = (byte)(0x40 | type);
    message[1] = (byte)(length>>8);
    message[2] = (byte)(length&0xFF);
    if (payload!=null) System.arraycopy(payload, 0, message, 3, length);
    try {
      this.out.write(message);
    } catch (IOException e) {
      e.printStackTrace();
      System.exit(0);
    }
  }
  
  /** Sets the mapping from keyboard numbers to player numbers as allocated by the server. Calling this method starts or stops the sending of keyboard events to the server.
   * @param keyboardNumber a number from '0' to '5' indicating for which set of keys you wish to set the mapping.
   * @param playerNumber the number from '0' to '5' allocated by the server, or '-1' to stop sending keyboard events for these keys.
   */
  public void setPlayerMapping(int keyboardNumber, int playerNumber) {
    this.pNums[keyboardNumber] = playerNumber;
  }
  
  /** Sets the 'msPerWord' value. This number encodes the frequency with which the game server sends keyboard information to the clients. It is the number of milliseconds between updates. */
  public void setMSPerWord(int msPerWord) {
    if (msPerWord>1000) throw new IllegalArgumentException(
      "Server must send at least one word of key data every 1000ms"
    );
    this.msPerWord = msPerWord;
  }
  
  /** Sets the NetworkListener whose 'processExtendedMessage()' method is called by 'getKeyData()'. Any previous listener is removed.
   * @param listener the new NetworkListener, or 'null' to stop listening.
   */
  public void setListener(NetworkListener listener) {
    this.listener = listener;
  }
  
  /* Override things in Controller. */
  
  /** Returns a bitmask for every millisecond that has passed since this method was last called. Bit 'n' in a mask indicates whether key 'n' was pressed in that millisecond.
   * <p>This implementation reads as much data as it can from the server, up to a maximum of one second's worth, and returns it. The server sends a 32-bit word for every 'msPerWord' milliseconds of game time. Each is replicated 'msPerWord' times to make a word for every millisecond as required.
   * <p>This method is not re-entrant.
   * <p>This method has a rather abrupt behaviour in the face of an IOException: it calls 'System.exit(0)'. This should probably be fixed.
   */
  public int[] getKeyData() {
    int used = 0;
    try {
      while (
        used==0 ||
        (this.in.available()>=4 && used<=BUF.length-this.msPerWord)
      ) {
  	int mask = this.in.read();
  	if ((mask&(1<<6))!=0) { // Extended message.
  	  final int length = (this.in.read()<<8) | this.in.read();
          if (length<0) throw new IOException(
            "Socket closed (during extended message length)"
          );
  	  final byte[] payload = new byte[length];
  	  int got = 0;
  	  while (got<length) {
            final int read = this.in.read(payload, got, length-got);
            if (read==-1) throw new IOException(
              "Socket closed (during payload)"
            );
            got += read;
          }
  	  if (this.listener!=null)
            this.listener.processExtendedMessage(mask&0x3F, payload);
  	} else { // Simple message.
  	  mask = (mask<<8) | this.in.read();
  	  mask = (mask<<8) | this.in.read();
  	  mask = (mask<<8) | this.in.read();
          if ((mask&(1<<31))!=0) mask ^= 0xC0000000;
  	  for (int i=0; i<this.msPerWord; i++) BUF[used++] = mask;
  	}
      }
    } catch (IOException e) {
      e.printStackTrace();
      System.exit(0);
    }
    final int[] ans = new int[used];
    System.arraycopy(BUF, 0, ans, 0, used);
    return ans;
  }
  
  /* Implement things in KeyListener. */
  
  /** Called by the AWT when a key is pressed. */
  public void keyPressed(KeyEvent e) {
    this.send(0x00, this.decode(e));
  }
  
  /** Called by the AWT when a key is released. */
  public void keyReleased(KeyEvent e) {
    this.send(0x80, this.decode(e));
  }
  
  /** Called by the AWT when a character is typed. We don't care. */
  public void keyTyped(KeyEvent e) {}

  /* Private. */
  
  /** Sends a key event to the server.
   * @param control '0x00' for key down or '0x80' for key up.
   * @param code the key code, from '0' to '30'.
   */
  private synchronized void send(int control, int code) {
    if (code==-1) return;
    try {
      if (code==30) { out.write(control | 0x07); return; }
      int pNum = this.pNums[code/5]; if (pNum==-1) return;
      this.out.write(control | (pNum<<3) | (code%5));
    } catch (IOException e) {
      e.printStackTrace();
      System.exit(0);
    }
  }
  
  private int[] pNums;
  private NetworkListener listener;
  private InputStream in;
  private OutputStream out;
  private int msPerWord;
  
  private static int[] BUF = new int[1000];

  /* Test code. */

  public static void main(String[] args) {
    if (args.length!=0) throw new IllegalArgumentException(
      "Syntax: java org.sc3d.apt.jrider.v1.NetworkController"
    );
    throw new RuntimeException("Not yet implemented");
  }
}
