package org.sc3d.apt.jrider.v1;

/** A RigidBody represents the position, orientation, momentum and angular momentum of a rigid body in 3D space, and provides methods for applying forces on it and integrating its equations of motion. */
public class RigidBody extends CollisionSphere.Body {
  /** Constructs a stationary RigidBody with the specified mass, moments of inertia, position and orientation.
   * @param mass the mass of the body in kg.
   * @param ixb the moment of inertia of the body about its x-axis in kgm2.
   * @param iyb the moment of inertia of the body about its y-axis.
   * @param izb the moment of inertia of the body about its z-axis.
   * @param t the position of the body (velocity is ignored).
   * @param o the orientation of the body (angular velocity is ignored).
   */
  public RigidBody(
    int mass,
    int ixb, int iyb, int izb,
    Trajectory t, Orientation o
  ) {
    this.mass = mass;
    this.ixb = ixb; this.iyb = iyb; this.izb = izb;
    this.x = t.x; this.y = t.y; this.z = t.z;
    this.xx = o.xx; this.xy = o.xy; this.xz = o.xz;
    this.yx = o.yx; this.yy = o.yy; this.yz = o.yz;
    this.zx = o.zx; this.zy = o.zy; this.zz = o.zz;
    this.px = this.py = this.pz = 0;
    this.jx = this.jy = this.jz = 0;
    this.t = null; this.o = null;
  }

  /** Constructs a stationary RigidBody with the specified mass, moments of inertia and position, and with its axes aligned with those of the world.
   * @param mass the mass of the body in kg.
   * @param ixb the moment of inertia of the body about its x-axis in kgm2.
   * @param iyb the moment of inertia of the body about its y-axis.
   * @param izb the moment of inertia of the body about its z-axis.
   * @param x the x-ccordinate (east) of the body, in units such that the size of the landscape is '1&lt;&lt;32' (so 1m is '1&lt;&lt;22').
   * @param y the y-coordinate (north) of the body.
   * @param z the z-coordinate (up) of the body.
   */
  public RigidBody(
    int mass,
    int ixb, int iyb, int izb,
    int x, int y, int z
  ) {
    this(mass, ixb, iyb, izb, new Trajectory(x, y, z), Orientation.ID);
  }
  
  /* New API. */
  
  /** Returns the position and velocity of the centre of mass of this RigidBody. The velocity is calculated from the momentum and the mass. */
  public Trajectory getTrajectory() {
    if (this.t==null) {
      this.t = new Trajectory(
        this.x, this.y, this.z,
        this.px/this.mass, this.py/this.mass, this.pz/this.mass
      );
    }
    return this.t;
  }
  
  /** Returns the orientation and angular velocity of this RigidBody. The angular velocity is calculated from the angular momentum, the orientation, and the moments of inertia. The calculation is a little expensive (it takes about 30 lines of source code). The result is cached, and so the calculation will not be repeated if this method is called several times. However, the cached value is discarded whenever any of 'applyForce()', 'applyTorque()' or 'tick()' are called. */
  public Orientation getOrientation() {
    if (this.o==null) {
      // Work out angular velocity in the body's own frame.
      int wxb = (int)((
        this.jx*(long)this.xx +
        this.jy*(long)this.xy +
        this.jz*(long)this.xz
      ) / (this.ixb<<16));
       int wyb = (int)((
        this.jx*(long)this.yx +
        this.jy*(long)this.yy +
        this.jz*(long)this.yz
      ) / (this.iyb<<16));
      int wzb = (int)((
        this.jx*(long)this.zx +
        this.jy*(long)this.zy +
        this.jz*(long)this.zz
      ) / (this.izb<<16));
      // Hack to prevent arithmetic overflow.
      if (wxb>(1<<16)) wxb = 1<<16; else if (wxb<-1<<16) wxb = -1<<16;
      if (wyb>(1<<16)) wyb = 1<<16; else if (wyb<-1<<16) wyb = -1<<16;
      if (wzb>(1<<16)) wzb = 1<<16; else if (wzb<-1<<16) wzb = -1<<16;
      // Transform the angular velocity into the world's frame, and return it.
      this.o = new Orientation(
        this.xx, this.xy, this.xz,
        this.yx, this.yy, this.yz,
        this.zx, this.zy, this.zz,
        (wxb*this.xx + wyb*this.yx + wzb*this.zx) >> 14,
        (wxb*this.xy + wyb*this.yy + wzb*this.zy) >> 14,
        (wxb*this.xz + wyb*this.yz + wzb*this.zz) >> 14
      );
    }
    return this.o;
  }
  
  /** Negates both the x- and y-axes of this RigidBody. In other words, it rotates the body 180 degrees about its own z-axis. */
  public void flipXY() {
    this.xx = -this.xx; this.xy = -this.xy; this.xz = -this.xz;
    this.yx = -this.yx; this.yy = -this.yy; this.yz = -this.yz;
    // Invalidate the cached Orientation.
    this.o = null;
  }
  
  /** Applies the specified torque for 1ms.
   * @param tx the x-component of the torque, in units of 1Nm (so 1 kilogram landscape squared per millisecond squared is '1&lt;&lt;40).
   * @param ty the y-component of the torque.
   * @param tz the z-component of the torque.
   */
  public void applyTorque(int tx, int ty, int tz) {
    this.jx += tx; this.jy += ty; this.jz += tz;
    // Invalidate the cached Orientation.
    this.o = null;
  }
  
  /** Accelerates this RigidBody, as if by a gravitational field.
   * @param ax the x-component of the acceleration, in units such that 1m/s2 is '1&lt;&lt;2' (so a landscape per millisecond squared is '1&lt;&lt;32').
   * @param ay the y-component of the acceleration.
   * @param az the z-component of the acceleration.
   */
  public void accelerate(int ax, int ay, int az) {
    this.px += this.mass*ax; this.py += this.mass*ay; this.pz += this.mass*az;
    // Invalidate the cached Trajectory.
    this.t = null;
  }
  
  /** Works out the position and orientation of this RigidBody 1ms later, by integrating its equations of motion. This method ends by tweaking the axes to make sure they have the right length and are orthogonal. */
  public void tick() {
    // Work out the velocity and add it to the position.
    final Trajectory t = this.getTrajectory();
    this.x += t.vx; this.y += t.vy; this.z += t.vz;
    // Work out the angular velocity and rotate the axes.
    final Orientation o = this.getOrientation();
    this.xx += (o.wy*o.xz - o.wz*o.xy) >> 18;
    this.xy += (o.wz*o.xx - o.wx*o.xz) >> 18;
    this.xz += (o.wx*o.xy - o.wy*o.xx) >> 18;
    this.yx += (o.wy*o.yz - o.wz*o.yy) >> 18;
    this.yy += (o.wz*o.yx - o.wx*o.yz) >> 18;
    this.yz += (o.wx*o.yy - o.wy*o.yx) >> 18;
    this.zx += (o.wy*o.zz - o.wz*o.zy) >> 18;
    this.zy += (o.wz*o.zx - o.wx*o.zz) >> 18;
    this.zz += (o.wx*o.zy - o.wy*o.zx) >> 18;
    // Correct the length of the axes.
    int f;
    f = (this.xx*this.xx + this.xy*this.xy + this.xz*this.xz - (1<<28)) >> 12;
    this.xx -= (this.xx*f) >> 17;
    this.xy -= (this.xy*f) >> 17;
    this.xz -= (this.xz*f) >> 17;
    f = (this.yx*this.yx + this.yy*this.yy + this.yz*this.yz - (1<<28)) >> 12;
    this.yx -= (this.yx*f) >> 17;
    this.yy -= (this.yy*f) >> 17;
    this.yz -= (this.yz*f) >> 17;
    f = (this.zx*this.zx + this.zy*this.zy + this.zz*this.zz - (1<<28)) >> 12;
    this.zx -= (this.zx*f) >> 17;
    this.zy -= (this.zy*f) >> 17;
    this.zz -= (this.zz*f) >> 17;
    // Correct the angles between the axes.
    f = (this.yx*this.zx + this.yy*this.zy + this.yz*this.zz) >> 12;
    this.yx -= (this.zx*f) >> 17; this.zx -= (this.yx*f) >> 17;
    this.yy -= (this.zy*f) >> 17; this.zy -= (this.yy*f) >> 17;
    this.yz -= (this.zz*f) >> 17; this.zz -= (this.yz*f) >> 17;
    f = (this.zx*this.xx + this.zy*this.xy + this.zz*this.xz) >> 12;
    this.zx -= (this.xx*f) >> 17; this.xx -= (this.zx*f) >> 17;
    this.zy -= (this.xy*f) >> 17; this.xy -= (this.zy*f) >> 17;
    this.zz -= (this.xz*f) >> 17; this.xz -= (this.zz*f) >> 17;
    f = (this.xx*this.yx + this.xy*this.yy + this.xz*this.yz) >> 12;
    this.xx -= (this.yx*f) >> 17; this.yx -= (this.xx*f) >> 17;
    this.xy -= (this.yy*f) >> 17; this.yy -= (this.xy*f) >> 17;
    this.xz -= (this.yz*f) >> 17; this.yz -= (this.xz*f) >> 17;
    // Incur some losses.
    this.jx -= this.jx>>15; this.jy -= this.jy>>15; this.jz -= this.jz>>15;
    // Invalidate the cached Trajectory and Orientation.
    this.t = null; this.o = null;
  }
  
  /* Override things in Body. */
  
  /** Applies the specified force at the specified point for 1ms.
   * @param fx the x-component of the force, in units such that 1N is '1&lt;&lt;2' (so 1 kilogram landscape per millisecond squared is '1&lt;&lt;32').
   * @param fy the y-component of the force.
   * @param fz the z-component of the force.
   * @param x the x-coordinate of the point at which the force is applied, in units such that the size of the landscape is '1&lt;&lt;32' (so 1m is '1&lt;&lt;22').
   * @param y the y-coordinate of the point at which the force is applied.
   * @param z the z-coordinate of the point at which the force is applied.
   */
  public void applyForce(int fx, int fy, int fz, int x, int y, int z) {
    // Add to the momentum.
    this.px += fx; this.py += fy; this.pz += fz;
    // Add to the angular momentum.
    x -= this.x; y -= this.y; z -= this.z;
    this.jx += (int)((y*(long)fz - z*(long)fy) >> 24);
    this.jy += (int)((z*(long)fx - x*(long)fz) >> 24);
    this.jz += (int)((x*(long)fy - y*(long)fx) >> 24);
    // Invalidate the cached Trajectory and Orientation.
    this.t = null; this.o = null;
  }
  
  /* Private. */
  
  /** The mass of the body in units of 1kg. */
  private final int mass;
  
  /** The moment of inertia of the body about one of its axes, in units of '1kgm2'. */
  private final int ixb, iyb, izb;
  
  /** A coordinate, in units such that the size of the landscape is '1&lt;&lt;32' (so a metre is '1&lt;&lt;22'). */
  private int x, y, z;
  
  /** A component of the vector of length '1&lt;&lt;14' along the x-axis of this RigidBody. */
  private int xx, xy, xz;
  
  /** A component of the vector of length '1&lt;&lt;14' along the y-axis of this RigidBody. */
  private int yx, yy, yz;
  
  /** A component of the vector of length '1&lt;&lt;14' along the z-axis of this RigidBody. */
  private int zx, zy, zz;
  
  /** A component of the momentum, in units such that 1kgm/s is '1&lt;&lt;12' (so 1 kilogram landscape per millisecond is '1&lt;&lt;32'). */
  private int px, py, pz;
  
  /** A component of the angular momentum, in units such that 1kgm2/s is '1&lt;&lt;10' (so 1 kilogram landscape squared per millisecond is '1&lt;&lt;40'). */
  private int jx, jy, jz;
  
  /** A cache of the result of 'getTrajectory()'. */
  private Trajectory t;
  
  /** A cache of the result of getOrientation()'. */
  private Orientation o;
  
  /* Test code. */
  
  public static void main(String[] args) {
    // Make a RigidBody that represents a spinning car chassis.
    final RigidBody me = new RigidBody(300, 300, 75, 300, 0, 0, 0);
    me.applyTorque(-300<<10, 0, 300<<10);
    final Model m = Model.CARS[1];
    // Animate it spinning.
    final Camera c = new Camera(0, -1<<24, 0, 0<<16, 1<<16);
    final Lens lens = new Lens(256, 256, 128, 128);
    final SceneImage si = new SceneImage(lens, 512, 612, "Testing RigidBody");
    long time = System.currentTimeMillis();
    while (true) {
      Frame f = si.reset(c, null);
      m.project(me.getTrajectory(), me.getOrientation(), f);
      si.doFrame();
      long nextTime = System.currentTimeMillis();
      while (time<nextTime) {
        me.tick();
        time++;
      }
    }
  }
}
