/*
 * game.c:
 * Objects representing games.
 *
 * Copyright (c) 2003 Chris Lightfoot. All rights reserved.
 * Email: chris@ex-parrot.com; WWW: http://www.ex-parrot.com/~chris/
 *
 */

static const char rcsid[] = "$Id: game.c,v 1.5 2003/10/19 21:15:33 chris Exp chris $";

#include <assert.h>

#include <sys/types.h>
#include <stdbool.h>
#include <stdint.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <netinet/in.h>

#include <sys/socket.h>
#include <sys/time.h>

#include "game.h"
#include "peer.h"
#include "protocol.h"
#include "util.h"

/* struct game
 * Game object internals. */
struct game {
    char *g_trackname;
    uint8_t *g_checksum;
    size_t g_checksum_len;
    struct timeval g_next_tick;
    uint16_t g_timestep;
    
    int g_num_players;
    struct player {
        peer p_peer; /* connection to player */
        bool p_has_sent_checksum;
        enum { p_notpresent, p_negotiating, p_playing } p_state;
        /* car parameters */
        uint8_t *p_carparams;
        size_t p_carparams_len;
        /* controls */
        bool p_key_map[MAX_KEYS]; /* true means down */
    } g_players[MAX_PLAYERS];
};

#define DEFAULT_TIMESTEP        10

/* game_new TRACK TIMESTEP
 * Construct a new game object for the given TRACK name and TIMESTEP. */
game game_new(const char *track, const uint16_t timestep) {
    game g;
    alloc_struct(game, g);

    g->g_trackname = xstrdup(track);
    g->g_timestep = timestep ? timestep : DEFAULT_TIMESTEP;

    gettimeofday(&g->g_next_tick, NULL);
    add_milliseconds_to_timeval(&g->g_next_tick, g->g_timestep);
    
    return g;
}

#ifndef SHUT_RDWR
#   define SHUT_RDWR    2
#endif

/* send_message GAME PLAYER MESSAGE
 * Send MESSAGE to the named PLAYER, or all players if PLAYER is -1, in GAME.
 * Returns 1 on success or 0 otherwise. */
static int send_message(game g, const int player, const struct sc_message *msg) {
    static uint8_t *buf;
    static size_t buflen;
    size_t pktlen;

    buf = scm_to_wire(msg, &pktlen, buf, &buflen);
    if (!pktlen)
        return 0;

    if (msg->scm_extended) {
        fprintf(stderr, "send_message: to player %d\n", player);
        scm_dump(stderr, "send_message- ", msg);
    }
    
    if (player == -1) {
        int i;
        for (i = 0; i < MAX_PLAYERS; ++i)
            if (g->g_players[i].p_peer) {
                int j = 0;
                /* Don't send any message to any peer more than once. */
                for (j = 0; j < MAX_PLAYERS; ++j)
                    if (g->g_players[i].p_peer == g->g_players[j].p_peer)
                        break;
                if (j == i && peer_write(g->g_players[i].p_peer, buf, pktlen) == -1)
                    return 0;
            }
    } else if (g->g_players[player].p_peer)
        if (peer_write(g->g_players[player].p_peer, buf, pktlen) == -1)
            return 0;

    return 1;
}

/* broadcast_message GAME MESSAGE
 * Send MESSAGE to all connected players in GAME. */
#define broadcast_message(g, m) send_message((g), -1, (m))

/* disconnect_player GAME PLAYER
 * Disconnect PLAYER from GAME, notifying other players of this news. Note
 * that this will in fact disconnect all players from the same peer. XXX make
 * this neater. */
static void disconnect_player(game g, const int player) {
    if (g->g_players[player].p_state != p_notpresent) {
        struct sc_message scm;
        struct player *P;
        peer p;
        int i;

        P = g->g_players + player;
        
        p = g->g_players[player].p_peer;

        for (i = 0; i < MAX_PLAYERS; ++i)
            if (g->g_players[i].p_peer == p) {
                g->g_players[player].p_state = p_notpresent;
                --g->g_num_players;
             
                /* Send player leaves message. */
                scm.scm_extended = 1;
                scm.scm_type = sc_player_leaves;
                scm.scm_player_num = player;
                broadcast_message(g, &scm);

                xfree(g->g_players[player].p_carparams);
                g->g_players[player].p_peer = NULL;
            }
        
        peer_delete(p);
    }
}

/* game_delete GAME
 * Free memory associated with GAME; also disconnect any remaining users. */
void game_delete(game g) {
    int i;
    if (!g) return;
    for (i = 0; i < MAX_PLAYERS; ++i)
        disconnect_player(g, i);
    xfree(g->g_trackname);
    xfree(g->g_checksum);
    xfree(g);
}

/* game_add_player GAME PEER
 * Add the connected PEER to the GAME, returning the player number they have
 * been assigned, -1 on failure, or -2 if there are no more players. In the
 * latter case, a message will have been returned to the peer to this
 * effect. */
int game_add_player(game g, peer p) {
    struct sc_message scm = {0};
    scm.scm_extended = 1;
fprintf(stderr, "game_add_player: peer %s\n", peer_string(p));
    if (g->g_num_players >= MAX_PLAYERS) {
        static uint8_t *buf;
        static size_t buflen;
        size_t pktlen;
#define GAME_FULL_MESSAGE   "Sorry, this game is full"
        scm.scm_type = sc_game_full;
        scm.scm_string = GAME_FULL_MESSAGE;
        scm.scm_string_len = sizeof(GAME_FULL_MESSAGE) - 1;

        /* can't use send_message, since this is not a player... */
        buf = scm_to_wire(&scm, &pktlen, buf, &buflen);
fprintf(stderr, "game_add_player: %s\n", peer_string(p));
scm_dump(stderr, "game_add_player- ", &scm);
        if (!pktlen || peer_write(p, buf, pktlen) == -1);
            return -1;

        return -2;
#undef GAME_FULL_MESSAGE
    } else {
        int assigned = -1;
        bool failed = 0;
        int i;
        for (i = 0; i < MAX_PLAYERS; ++i) {
            if (g->g_players[i].p_state == p_notpresent) {
                struct player pz = {0};
                int j;
                
                assigned = i;
                
                g->g_players[i] = pz;
                g->g_players[i].p_state = p_negotiating;
                g->g_players[i].p_peer = p;
                ++g->g_num_players;

                /* Tell the client which player number they've been assigned. */
                scm.scm_type = sc_assign_player;
                scm.scm_player_num = i;
                if (!send_message(g, i, &scm)) {
                    failed = 1;
                    break;
                }

                /* Also need to tell this player the parameters of all existing
                 * cars. */
                scm.scm_type = sc_set_car_parameters;
                for (j = 0; j < MAX_PLAYERS; ++j) {
                    if (g->g_players[i].p_state == p_negotiating
                        || !g->g_players[i].p_carparams)
                        continue;

                    scm.scm_player_num = j;
                    scm.scm_string = g->g_players[j].p_carparams;
                    scm.scm_string_len = g->g_players[j].p_carparams_len;

                    if (!send_message(g, i, &scm)) {
                        failed = 1;
                        break;
                    }
                }

                if (failed)
                    break;

                /* And the timestep in the game. */
                scm.scm_type = sc_time_step;
                scm.scm_timestep = g->g_timestep;
                if (!send_message(g, i, &scm)) {
                    failed = 1;
                    break;
                }
                
                break;
            }
        }
        assert(assigned != -1);

        if (failed) {
            /* Failed means a network error when we were talking to the peer.
             * This call will cause all the players associated with that
             * connection to be dropped, but if we can't talk to them, that's
             * fine. */
            disconnect_player(g, assigned);
            return -1;
        }
fprintf(stderr, "game_add_player: added player %d\n", assigned);

        return assigned;
    }
}

/* game_pre_select GAME N READFDS WRITEFDS EXCEPTFDS TIMEOUT
 * Set up file descriptor masks for all connected peers. Also ensure that the
 * timeout will complete before our next tick is due. */
void game_pre_select(game g, int *n, fd_set *rd, fd_set *wr, fd_set *ex, struct timeval *tv) {
    int i;
    struct timeval deadline, now;

    /* Move the timeout closer if it is too long. */
    deadline = g->g_next_tick;
    gettimeofday(&now, NULL);
    subtract_timevals(&deadline, &now);
    if (compare_timevals(&deadline, tv) < 0)
        *tv = deadline;
    if (tv->tv_sec < 0 || tv->tv_usec < 0) {
        tv->tv_sec = 0;
        tv->tv_usec = 0;
    }

//fprintf(stderr, "timeout is %d.%06d\n", (int)tv->tv_sec, (int)tv->tv_usec);
    
    /* Per-peer select handling. */
    for (i = 0; i < MAX_PLAYERS; ++i)
        if (g->g_players[i].p_peer)
            peer_pre_select(g->g_players[i].p_peer, n, rd, wr, ex);
}

/* process_client_message GAME PEER MESSAGE
 * Process a single MESSAGE from a connected PEER in GAME, returning 0 if it
 * was processed successfully, or -1 otherwise. */
static int process_client_message(game g, peer p, struct cs_message *csm) {
    struct player *P;
#define PLAYER_CHECKS(g, p, csm)   ((csm)->csm_player_num < MAX_PLAYERS && (g)->g_players[csm->csm_player_num].p_peer == (p))
    if (!(csm->csm_extended)) {
        /* Simple message about a single key being pressed/released. */
        if (!PLAYER_CHECKS(g, p, csm)
            && !(csm->csm_player_num == 0 && csm->csm_key == key_escape)) /* client will send escape keypresses as if from player 0 */
            return -1;
        
        P = g->g_players + csm->csm_player_num;
        
        if (csm->csm_key >= MAX_KEYS || csm->csm_key < 0
            || csm->csm_key == key_reserved_1 || csm->csm_key == key_reserved_2)
            return -1;                          /* invalid keys */
        P->p_key_map[csm->csm_key] = !csm->csm_key_up;
    } else {
        /* Extended message. */
        struct sc_message scm = {0};
        int i;

        scm.scm_extended = 1;
        
        switch (csm->csm_type) {
            case cs_chat_message:
                /* Chat message. Pass it on to all players. */
                scm.scm_type = sc_chat_message;
                scm.scm_string = csm->csm_string;
                broadcast_message(g, &scm);     /* XXX how should we deal with errors here? error flag in player, probably */
                break;

            case cs_claim_player:
                /* Claim another player. */
                game_add_player(g, p);
                break;

            case cs_choose_car_parameters:
                /* Set parameters of car. */
                if (!PLAYER_CHECKS(g, p, csm))
                    return -1;

                P = g->g_players + csm->csm_player_num;
                P->p_carparams = xrealloc(P->p_carparams, P->p_carparams_len = csm->csm_string_len);
                memcpy(P->p_carparams, csm->csm_string, csm->csm_string_len);
                
                /* XXX move this into a separate function, call also from add player */
                scm.scm_type = sc_set_car_parameters;
                scm.scm_player_num = csm->csm_player_num;
                scm.scm_string = csm->csm_string;
                scm.scm_string_len = csm->csm_string_len;

                broadcast_message(g, &scm);
                break;

            case cs_send_checksums:
                /* Player sends opaque checksum. If they are the first to do
                 * so, record this as the game checksum. Otherwise check that
                 * the checksum they've sent equals the game checksum, and
                 * disconnect them otherwise. */
                if (!g->g_checksum) {
                    g->g_checksum = csm->csm_string;
                    g->g_checksum_len = csm->csm_string_len;
                    csm->csm_string = NULL;
                } else {
                    if (g->g_checksum_len != csm->csm_string_len || memcmp(g->g_checksum, csm->csm_string, csm->csm_string_len))
                        return -1;
                    /* yuk. just set the has_sent_checksum flag for all players
                     * from this peer. */
                    for (i = 0; i < MAX_PLAYERS; ++i)
                        if (g->g_players[i].p_peer == p)
                            g->g_players[i].p_has_sent_checksum = 1;
                }
                break;

            case cs_list_tracks:
            case cs_choose_track:
                /* Game (track) selection. Too late for these; just ignore them. */
                break;
             
            default:
                /* Unknown packet. Error. */
                return -1;
        }
    }
#undef PLAYER_CHECKS
    return 0;
}

/* receive_packets GAME PLAYER
 * Read and process packets from PLAYER in GAME, updating the game state to
 * reflect any changes. Also passes on relevant information to other players.
 * Note that because a given peer may have more than one player, this may
 * update the state of players other than PLAYER. Returns 0 on success or
 * -1 in the case of a protocol error which should result in the player's
 * disconnection. */
static int receive_packets(game g, int player) {
    uint8_t *buf, *p;
    size_t len;
    int npkts = 0, n;
    if (!g->g_players[player].p_peer) return -1;
    p = buf = peer_get_data(g->g_players[player].p_peer, &len);
    do {
        struct cs_message csm;
        
        n = csm_from_wire(&csm, p, len - (p - buf));
        if (n == -1)
            return -1;
        else if (n == 0)
            break;
        
        ++npkts;
        p += n;
/*
        fprintf(stderr, "(game)receive_packets: %s\n", peer_string(g->g_players[player].p_peer));
        cs_dump(stderr, "(game)receive_packets- ", &csm);
*/

        if (process_client_message(g, g->g_players[player].p_peer, &csm) == -1)
            return -1;
        
        csm_free(&csm);
    } while (n > 0);
    peer_consume(g->g_players[player].p_peer, p - buf);
    return 0;
}

int broadcast_key_states(game g) {
    struct sc_message scm = {0};
    int i, j;
    for (i = 0; i < MAX_PLAYERS; ++i)
        for (j = 0; j < MAX_KEYS; ++j)
            scm.scm_player_key_map[i][j] = g->g_players[i].p_key_map[j];
    return broadcast_message(g, &scm);
}

/* game_post_select GAME READFDS WRITEFDS EXCEPTFDS
 * Post-select handling for the GAME. Also sends out next-tick information
 * if the time is right. Return 0 on success, or -1 on a serious error which
 * means that the game should be shut down. */
int game_post_select(game g, fd_set *rd, fd_set *wr, fd_set *ex) {
    struct timeval now;
    int i;

    /* allow peers to read data into their buffers, disconnecting any which
     * have fallen over. */
    for (i = 0; i < MAX_PLAYERS; ++i) {
        int n;
        struct player *P;
        P = g->g_players + i;
        if (!P || P->p_state == p_notpresent || !P->p_peer) continue;
        n = peer_post_select(P->p_peer, rd, wr, ex);
        if (n <= 0 || peer_error(P->p_peer)) {
            /* XXX where should we log these, and how? */
            fprintf(stderr, "game_post_select: peer_post_select: player %d: %s; dropping client\n", i, n == -1 ? strerror(peer_error(P->p_peer)) : "closed connection");
            disconnect_player(g, i);
        }
        
        if (receive_packets(g, i) == -1) {
            fprintf(stderr, "game_post_select: receive_packets: player %d: protocol error; dropping client\n", i);
            disconnect_player(g, i);
        }

        if (P->p_state == p_negotiating && P->p_has_sent_checksum && P->p_carparams)
            P->p_state = p_playing;
    }

    gettimeofday(&now, NULL);
//fprintf(stderr, "now = %d.%06d; deadline = %d.%06d\n", (int)now.tv_sec, (int)now.tv_usec, (int)g->g_next_tick.tv_sec, (int)g->g_next_tick.tv_usec);
    while (compare_timevals(&now, &g->g_next_tick) > 0) {
//fprintf(stderr, "time for next tick...\n");
        broadcast_key_states(g);
        add_milliseconds_to_timeval(&g->g_next_tick,  g->g_timestep);
        gettimeofday(&now, NULL);
    }

    return 0;
}

/* game_send_info GAME PEER
 * Send information about this GAME to the PEER. Returns 1 on success or 0 on
 * failure. */
int game_send_info(game g, peer p) {
    struct sc_message scm = {0};
    static uint8_t *buf;
    static size_t buflen;
    size_t pktlen;

    scm.scm_extended = 1;
    scm.scm_type = sc_track_info;
    scm.scm_string = g->g_trackname;
    scm.scm_player_num = g->g_num_players;

    buf = scm_to_wire(&scm, &pktlen, buf, &buflen);
fprintf(stderr, "game_send_info: %s\n", peer_string(p));
scm_dump(stderr, "game_send_info- ", &scm);
    if (!pktlen || peer_write(p, buf, pktlen) == -1)
        return 0;

    return 1;
}

/* game_trackname GAME
 * Return this game's track name. */
char *game_trackname(game g) {
    return g->g_trackname;
}
