package org.sc3d.apt.screen.v4;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.util.Hashtable;

/** A Screen is a frame buffer to draw on, whose contents are displayed on a Canvas, which can be displayed in a window like any other Component. A Screen also tracks keyboard and mouse events. */
public class Screen extends Canvas {
  /** Constructs a Screen of the specified size. */
  public Screen(int width, int height) {
    this.width = width; this.height = height;
    this.setSize(width,height);
    // Make a MemoryImageSource.
    this.pixels = new int[width*height];
    this.source = new MemoryImageSource(
      width,
      height,
      new DirectColorModel(24, 0xff0000, 0x00ff00, 0x0000ff),
      this.pixels,
      0,
      width
    );
    this.source.setAnimated(true);
    // Initialise the keyboard and mouse state.
    this.keysPressed = new Hashtable();
    this.mouseX = this.mouseY = 0;
    // Add the key and mouse listeners.
    this.addKeyListener(new KeyAdapter() {
      public void keyPressed(KeyEvent e) {
        Integer key = new Integer(e.getKeyCode());
        Screen.this.keysPressed.put(key, key);
      }
      public void keyReleased(KeyEvent e) {
        Integer key = new Integer(e.getKeyCode());
        Screen.this.keysPressed.remove(key);
      }
    });
    this.addMouseMotionListener(new MouseMotionAdapter() {
      public void mouseDragged(MouseEvent e) { this.mouseMoved(e); }
      public void mouseMoved(MouseEvent e) {
        Screen.this.mouseX = e.getX();
        Screen.this.mouseY = e.getY();
      }
    });
  }
  
  /** Constructs a Screen of the specified size, and displays it in a window with the specified title, which when closed calls 'System.exit(0)'. */
  public Screen(int width, int height, String title) {
    this(width, height);
    Frame frame = new Frame(title);
    frame.addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e) { System.exit(0); }
    });
    frame.add(this);
    frame.pack();
    frame.setVisible(true);
  }
  
  /* New API. */
  
  public int getWidth() { return this.width; }
  public int getHeight() { return this.height; }
  
  /** Returns a reference to the frame buffer. The array is read left to right, top to bottom, with one pixel per entry, in the default RGB colour model. The top 8 bits are ignored - all pixels are opaque. */
  public int[] getPixels() { return this.pixels; }
  
  /** Updates the window to reflect the latest state of the frame buffer. */
  public void doFrame() { this.source.newPixels(); }
  
  /** Reads the state of a key.
   * @param keyCode one of the KeyEvent.VK_XXX codes. */
  public boolean readKey(int keyCode) {
    return this.keysPressed.contains(new Integer(keyCode));
  }
  
  /** Reads the x-coordinate of the mouse. */
  public int getMouseX() { return this.mouseX; }
  /** Reads the y-coordinate of the mouse. */
  public int getMouseY() { return this.mouseY; }
  
  /** As 'n' increases from '0', this method returns a sequence of easily distinguishable colours. */
  public static int sequenceColour(int n) {
    int ans = 0;
    for (int i=0; i<8; i++) {
      ans <<= 1;
      if ((n&1)!=0) ans |= 1<<16;
      if ((n&2)!=0) ans |= 1<<8;
      if ((n&4)!=0) ans |= 1<<0;
      n >>>= 3;
    }
    return ans;
  }

  /* Override things in Canvas. */
  
  /** Called by the AWT to repair damaged pixels. */
  public void paint(Graphics g) {
    if (this.image==null) this.image = this.createImage(this.source);
    if (this.image!=null) g.drawImage(this.image, 0, 0, this);
  }
  
  /** Called by the AWT in response to 'repaint()' calls. */
  public void update(Graphics g) { this.paint(g); }
  
  /* Private. */
  
  private int width, height;
  private int[] pixels;
  private MemoryImageSource source;
  private Image image;
  private Hashtable keysPressed;
  private int mouseX, mouseY;
  
  /* Test code. */
  
  public static void main(String[] args) {
    final int W=640, H=480;
    final Screen me = new Screen(W, H, "Test");
    // Animation loop.
    Thread.currentThread().setPriority(Thread.MIN_PRIORITY);
    final int[] pixels = me.getPixels();
    while (true) {
      int time = (int)System.currentTimeMillis();
      int i=0;
      for (int y=0; y<H; y++) for (int x=0; x<W; x++) {
        int c1 = (5*time + x*x - y*y) >> 6;
        int c2 = (3*time + 2*x*y) >> 6;
        int c3 = (7*time + x*x + y*y) >> 6;
        pixels[i++] = ((c1&0xff)<<16) | ((c2&0xff)<<8) | (c3&0xff);
      }
      me.doFrame();
    }
  }
}
