package org.sc3d.apt.jrider.v1;

/** Represents a landscape in a form that allows efficient rendering. Also represents the course that a player must complete each lap. */
public class Landscape {
  /** Constructs a Landscape from an array of heights and colours. This is the most flexible way of constructing a Landscape.
   * @param log2size the logarithm to base 2 of the size of the landscape in squares.
   * @param heights an array with one entry for each square, in units such that the size of the landscape is '1<<32'.
   * @param colours an array with one entry for each square. This class shades each square of the landscape according to its gradient. In order for this to work, the palette must be defined so that bits 3 to 7 of a colour number choose one of 32 colours and bits 0 to 2 choose one of eight shades of that colour. The lighting algorithm only modifies bits 0 to 2. An exception is made for colours in the range '0xB8' to '0xBF', which are the colours used for water: they are left unchanged.
   * @param cambers an array of Cambers defining the course. 'cambers[0]' is the start line.
   */
  public Landscape(
    int log2size, int[] heights, byte[] colours, Camber[] cambers
  ) {
    this.log2size = log2size; this.cambers = cambers;
    int size = 1<<log2size;
    if (heights.length!=size*size)
      throw new IllegalArgumentException("heights.length is wrong");
    if (colours.length!=size*size)
      throw new IllegalArgumentException("colours.length is wrong");
    this.land = new int[log2size+1][];
    // Fill in the highest resolution by packing 'heights' and 'colours'.
    int[] l = this.land[log2size] = new int[size*size];
    for (int y=0; y<size; y++) for (int x=0; x<size; x++) {
      int index = x+size*y;
      l[BI[x]+2*BI[y]] = (heights[index]&~0xff) | (colours[index]&0xff);
    }
    // Fill in lower resolutions by taking maximum of 2x2 squares.
    for (int s=log2size-1; s>=0; s--) {
      int size2 = size; size /= 2;
      int[] l2 = l; l = this.land[s] = new int[size*size];
      for (int y=0; y<size; y++) for (int x=0; x<size; x++) {
        final int index = BI[x]+2*BI[y];
        l[index] = l2[4*index];
        if (l[index]<l2[4*index+1]) l[index] = l2[4*index+1];
        if (l[index]<l2[4*index+2]) l[index] = l2[4*index+2];
        if (l[index]<l2[4*index+3]) l[index] = l2[4*index+3];
      }
    }
    // Calculate illumination separately for each resolution.
    int seed = 874265341;
    for (int s=log2size; s>=0; s--) {
      size = 1<<s; l = this.land[s];
      int m = size-1, shift = 32-s-8;
      for (int y=0; y<size; y++) for (int x=0; x<size; x++) {
        int index = BI[x]+2*BI[y];
        if ((l[index]&0xF8)!=0xB8) { // It's not water.
          int nx = (l[BI[(x-1)&m]+2*BI[y]] - l[BI[(x+1)&m]+2*BI[y]]) >> shift;
          int ny = (l[BI[x]+2*BI[(y-1)&m]] - l[BI[x]+2*BI[(y+1)&m]]) >> shift;
          int ib2 = (nx*nx + ny*ny + (1<<16));
          l[index] &= ~7;
          seed *= 794835797;
          for (int j=0x20+(seed>>>27); (((ib2>>8)*j)>>8)*j<(1<<16); j+=0x20)
            l[index]++;
        }
      }
    }
  }
  
  /* New API. */
  
  /** Generates a Landscape from a random seed, using the facilities provided by LandGen and Track. This is the easiest way of constructing a Landscape.
   * @param seed a String used as a random seed.
   * @param log2size the logarithm to the base 2 of the number of squares along one side of the Landscape.
   */
  public static Landscape generate(String seed, int log2size) {
    System.out.print("Generating landscape...");
    final LandGen lg = new LandGen(new Random(seed+"Land"), log2size);
    System.out.print(" track...");
    final Track track = new Track(new Random(seed+"Track"), lg, 26);
    for (int i=0; i<20000; i++) track.improve();
    final Camber[] cambers = track.getCambers(-1, 24<<22);
    lg.drawRoad(cambers);
    System.out.print(" smoothing...");
    lg.smooth(); lg.smooth(); lg.smooth();
    System.out.print(" water...");
    int minZ = Integer.MAX_VALUE;
    for (int i=0; i<cambers.length; i++) {
      if (cambers[i].z<minZ) minZ = cambers[i].z;
    }
    lg.addWater(minZ-(1<<22));
    System.out.print(" trees...");
    lg.addTrees(new Random(seed+"Trees"));
    System.out.print(" lighting...");
    final Landscape ans =
      new Landscape(log2size, lg.getHeights(), lg.getColours(), cambers);
    System.out.println("\nFinished.");
    return ans;
  }
  
  /** Calculates a simple checksum of this Landscape. */
  public int getCheckSum() {
    final int[] l = this.land[this.log2size];
    int a = 0, b = 0;
    for (int i=0; i<l.length; i++) {
      a += l[i]; b += a;
    }
    return b;
  }
  
  /** Reads the height of the specified point. Only the highest-resolution representation  of the landscape is used. Heights are bi-linearly interpolated between neighbouring squares.
   * @param x the x-coordinate of the point (east) in units such that the size of the landscape is '1<<32'.
   * @param y the y-coordinate of the point (north).
   * @return the height, in units such that the size of the landscape is '1<<32'.
   */
  public int getHeight(int x, int y) {
    final int SIZE = 1<<this.log2size;
    final int SHIFT = 32-this.log2size, HSQ = 1<<SHIFT-1;
    final int[] L = this.land[this.log2size];
    int x0 = (x-HSQ)>>>SHIFT, x1 = (x+HSQ)>>>SHIFT;
    int y0 = (y-HSQ)>>>SHIFT, y1 = (y+HSQ)>>>SHIFT;
    int h0 = L[BI[x0]+2*BI[y0]]>>8, h1 = L[BI[x1]+2*BI[y0]]>>8;
    int h2 = L[BI[x0]+2*BI[y1]]>>8, h3 = L[BI[x1]+2*BI[y1]]>>8;
    int dx = (x-HSQ-(x0<<SHIFT))>>>(SHIFT-8);
    int dy = (y-HSQ-(y0<<SHIFT))>>>(SHIFT-8);
    h0 += (dx*(h1-h0) + 128) >> 8; h2 += (dx*(h3-h2) + 128) >> 8;
    return (h0<<8) + dy*(h2-h0);
  }
  
  /** Reads the colour of the specified point at the specified resolution.
   * @param x the x-coordinate of the point (east) in units such that the size of the landscape is '1<<32'.
   * @param y the y-coordinate of the point (north).
   * @param octave the logarithm to the base 2 of the number of squares per km.
   * @return the colour.
   */
  public byte getColour(int x, int y, int octave) {
    if (octave>this.log2size) octave = this.log2size;
    final int SHIFT = 32-octave;
    return (byte)this.land[octave][BI[x>>>SHIFT]+2*BI[y>>>SHIFT]];
  }
  
  /** Returns the Cambers passed to the constructor. */
  public Camber[] getCambers() { return this.cambers; }
  
  /** Draws a skid mark on the landscape, at all resolutions. Only squares with a colour in the range '0x10' to '0x17' (tarmac) will be affected. The colour will be changed to be in the range '0x08' to '0x0f'.
   * @param x the x-coordinate of the point (east) in units such that the size of the landscape is '1<<32'.
   * @param y the y-coordinate of the point (north).
   */
  public void drawSkidMark(int x, int y) {
    final int SHIFT = 32-this.log2size;
    int index = BI[x>>>SHIFT]+2*BI[y>>>SHIFT];
    for (int n = this.log2size; n>=0; n--) {
      final int[] l = this.land[n];
      final int old = l[index];
      if ((0xf8&old)!=0x10) return;
      l[index] ^= 0x10 ^ 0x08;
      if (old<l[index^1] || old<l[index^2] || old<l[index^3]) return;
      index >>= 2;
    }
  }
  
  /** Renders a vertical line of pixels of a picture of this Landscape as seen from the specified point. The previous contents of 'zbuf' and 'cbuf' are discarded, and they are then filled with distances and colours. Pixels higher up the strip (nearer the beginning of the array) are always more distant than those lower down. The sky is filled with colours '0xe0' to '0xff'.
   * @param x the x-coordinate of the viewer (east), in units such that the size of the landscape is '1<<32'.
   * @param y the y-coordinate of the viewer (north).
   * @param z the z-coordinate of the viewer (up).
   * @param xdw the x-component of the vector from the viewer to the bottom-most pixel of the strip, in units such that the distance from the viewer to the centre of the picture is '1<<16'.
   * @param ydw the y-component.
   * @param zdw the z-component.
   * @param zdv the z-component of the vector from pixel 'i' to pixel 'i-1' of the strip, in the same units as 'zdw'. This must be positive, and the x- and y-components are assumed to be zero, i.e. the camera is upright.
   * @param zbuf an array with one entry for each pixel. Entry '0' is at the top. When this method returns, each entry represents the distance to the part of the landscape that is visible in its pixel, in units such that the size of the landscape is '1<<16'.
   * @param cbuf an array with one entry for each pixel. When this method returns, this array is filled with colours.
   * @param detail roughly the number of squares to examine at each resolution. Values a bit smaller than the horizontal screen resolution are sensible.
   * @param seed a random seed to use for dithering, or '0' for no dithering. The effect of supplying a non-zero value here is to move 'x' and 'y' by a fraction of a pixel in a random direction. There is no effect on the CPU usage.
   */
  public void render(
    int x, int y, int z,
    int xdw, int ydw, int zdw,
    int zdv,
    int[] zbuf, byte[] cbuf,
    int detail,
    int seed
  ) {
    int v = zbuf.length - 1; // Index into 'zbuf' of next pixel.
    int w = 0; // Distance, as in 'zbuf'.
    int wzdv = w*zdv; // Should always hold.
    int octave = this.log2size+2, dw = 1<<16-octave; // Size of steps.
    int dwzdv = dw*zdv; // Should always hold.
    int dwxdw = dw*xdw, dwydw = dw*ydw, dwzdw = dw*zdw; // Should always hold.
    int wlimit = dw*detail*2; // Distance at which resolution drops.
    seed ^= 485643875*seed;
    final int ditherX = seed*1398475395, ditherY = seed*2028745485;
    while (w<(2<<16)) {
      final int res = octave<=this.log2size ? octave : this.log2size;
      final int shift=32-res;
      final int[] L = this.land[res];
      x += ditherX>>res+1; y += ditherY>>res+1;
      while (w<wlimit) {
        int hc = L[BI[x>>>shift]+2*BI[y>>>shift]];
        while (z<hc) {
          // Decrement 'v'.
          zbuf[v] = w; cbuf[v] = (byte)hc;
          v--; if (v<0) return;
          z += wzdv; dwzdw += dwzdv; // zdw += zdv;
        }
        // Increment 'w'.
        w += dw;
        wzdv += dwzdv;
        x += dwxdw; y += dwydw; z += dwzdw;
      }
      if (dwzdw>0 && z>this.land[0][0]) break;
      x -= ditherX>>res+1; y -= ditherY>>res+1;
      octave--; dw *= 2;
      dwzdv *= 2;
      dwxdw *= 2; dwydw *= 2; dwzdw *= 2;
      wlimit *= 2;
    }
    // Fill in the rest with sky.
    int l = ((xdw>>2)*(xdw>>2) + (ydw>>2)*(ydw>>2)) >> 14;
    if (l>(2<<14)) { zdw >>= 1; zdv >>= 1; l >>= 2; }
    int f = ((3<<14) - l) >> 1;
    for (int i=0; i<3; i++) f = (f * ((3<<14) - ((((f*f)>>14)*l)>>14))) >> 15;
    final int cdv = (zdv*f) >> 14;
    int c = (((zdw + (zbuf.length-1-v)*zdv)*f) >> 14) - (1<<16);
    for (; v>=0; v--) {
      zbuf[v] = (w<<8);
      cbuf[v] = (byte)(c>=0 ? 0xff : c<(-1<<16) ? 0xe0 : (c>>11));
      c += cdv;
    }
  }

  /* Private. */
  
  private int log2size;
  private Camber[] cambers;
  
  /** land[i] describes the Landscape as a grid with '1&lt;&lt;i' squares on a side. Each entry uses the top 24 bits for height (signed) and the bottom 8 bits for colour. Lower-resolution representations are calculated from higher-resolution representations by choosing the highest point in 2x2 squares. */
  private int[][] land;
  
  /** Look-up table. Entry 'n' gives the 'n'th number all of whose even-numbered bits are zero. The square at 'x', 'y' is stored in 'land' at index 'BI[x]+2*BI[y]'. This helps ensure nearby squares are stored at nearby indices, and improves cacheing behaviour. */
  private static final int[] BI = new int[0x1000];
  static { for (int i=1; i<0x1000; i++) BI[i] = (BI[i>>1]<<2) | (i&1); }
  
  /* Test code. */
  
  public static void main(String[] args) {
    if (args.length!=1) throw new IllegalArgumentException(
      "Syntax: java org.sc3d.apt.jrider.v1.Landscape <seed>"
    );
    Landscape me = Landscape.generate(args[0], 10);
    org.sc3d.apt.screen.v4.Screen screen =
      new org.sc3d.apt.screen.v4.Screen(512, 256, "Testing Landscape");
    int[] pixels = screen.getPixels();
    int[] zbuf = new int[256];
    byte[] cbuf = new byte[256];
    // Animation loop.
    Thread.currentThread().setPriority(Thread.MIN_PRIORITY);
    int px = 0, py = 0;
    while (true) {
//      System.gc();
      int pz = me.getHeight(px,py)+(1<<24);
      for (int u=1; u<512; u+=2) {
        me.render(
          px, py, pz,
          (u-256)<<8, 1<<16, -1<<15,
          1<<8,
          zbuf, cbuf,
          128,
          u
        );
        for (int v=0; v<256; v++)
          pixels[u-1+512*v] = pixels[u+512*v] = 0x010410 * (cbuf[v]&0xff);
      }
      screen.doFrame();
      py += 1<<22;
    }
  }
}
