package org.sc3d.apt.jrider.v1;

/** A Model is a 3D shape that can be used to represent a RigidBody on screen. */
public class Model {
  /** Constructs a Model consisting of the specified points, joined by the specified triangles.
   * @param xb the x-coordinates of the points, expressed in the frame of reference of the Model, in units such that the size of the landscape is '1&lt;&lt;20'.
   * @param yb the y-coordinates of the points.
   * @param zb the z-coordinates of the points.
   * @param faces an array with one entry for each triangular face. In each entry, the top 8 bits, the next 8 bits and the next 8 bits identify the three corners of the triangle, and the final 8 bits give the colour.
   */
  public Model(int[] xb, int[] yb, int[] zb, int[] faces) {
    this.xb = xb; this.yb = yb; this.zb = zb;
    this.faces = faces;
  }
  
  /* New API. */
  
  /** Constructs Faces representing this Model as it would appear at the specified position and orientation, viewed from 'f.camera' through 'f.lens', and merges them with 'f.faces'. */
  public void project(final Trajectory t, final Orientation o, final Frame f) {
    final Camera c = f.camera;
    final Lens l = f.lens;
    // Transform the object into the Camera's frame.
    final long x = (c.wy*(long)(t.x-c.x) - c.wx*(long)(t.y-c.y)) >> 16;
    final long y = (c.wx*(long)(t.x-c.x) + c.wy*(long)(t.y-c.y)) >> 16;
    final long z = t.z - c.z;
    final int xx = (c.wy*o.xx - c.wx*o.xy) >> 16;
    final int xy = (c.wx*o.xx + c.wy*o.xy) >> 16;
    final int yx = (c.wy*o.yx - c.wx*o.yy) >> 16;
    final int yy = (c.wx*o.yx + c.wy*o.yy) >> 16;
    final int zx = (c.wy*o.zx - c.wx*o.zy) >> 16;
    final int zy = (c.wx*o.zx + c.wy*o.zy) >> 16;
    final int xz = o.xz, yz = o.yz, zz = o.zz;
    // Project all the points.
    final int centreu = l.width << 7, centrev = l.height << 7;
    final int lensu = l.lensu << 8, lensv = l.lensv << 8;
    final int n = this.zb.length;
    final int[] u = new int[n], v = new int[n], w = new int[n];
    final int[] sv = new int[n]; // Shadow 'v'.
    for (int i=0; i<n; i++) {
      final int xbi = xb[i], ybi = this.yb[i], zbi = this.zb[i];
      final long xi = x + ((xbi*xx + ybi*yx + zbi*zx) >> 2);
      final long yi = y + ((xbi*xy + ybi*yy + zbi*zy) >> 2);
      final long zi = z + ((xbi*xz + ybi*yz + zbi*zz) >> 2);
      int szi = 0;
      if (f.land!=null) {
        final int sxi = t.x + ((xbi*o.xx + ybi*o.yx + zbi*o.zx) >> 2);
        final int syi = t.y + ((xbi*o.xy + ybi*o.yy + zbi*o.zy) >> 2);
        szi = f.land.getHeight(sxi, syi);
      }
      if (yi!=0) {
        u[i] = centreu + (int)((lensu*xi) / yi);
        v[i] = centrev - (int)((lensv*zi) / yi);
        sv[i] = centrev - (int)((lensv*(long)(szi-c.z)) / yi);
      }
      w[i] = (int)(yi >> 16);
    }
    // Construct the Faces.
    for (int i=0; i<this.faces.length; i++) {
      final int face = this.faces[i];
      final int n1 = (face>>24) & 0xff;
      final int n2 = (face>>16) & 0xff;
      final int n3 = (face>>8) & 0xff;
      // A triangle of the object.
      final Face ans = Triangle.make(
        u[n1], v[n1], w[n1],
        u[n2], v[n2], w[n2],
        u[n3], v[n3], w[n3],
        Face.ORDER_OBJECT,
        (byte)face,
        l.width, l.height
      );
      if (ans!=null) f.faces = ans.merge(f.faces);
      // A triangle of the shadow.
      final Face sAns = Triangle.make(
        u[n3], sv[n3], w[n3],
        u[n2], sv[n2], w[n2],
        u[n1], sv[n1], w[n1],
        Face.ORDER_SHADOW,
        (byte)0xff,
        l.width, l.height
      );
      if (sAns!=null) f.faces = sAns.merge(f.faces);
    }
  }

  /* Useful Models. */
  
  /** Cars in each of the six available colours. */
  public static final Model[] CARS = new Model[] {
    new Car(0x88), new Car(0x90), new Car(0x98),
    new Car(0xa0), new Car(0xa8), new Car(0xb0)
  };
  
  /** Cockpits in each of the six available colours. */
  public static final Model[] COCKPITS = new Model[] {
    new Cockpit(0x88), new Cockpit(0x90), new Cockpit(0x98),
    new Cockpit(0xa0), new Cockpit(0xa8), new Cockpit(0xb0)
  };
  
  /** A Model that looks like a wheel. */
  public static final Model WHEEL = new Model(
    new int[] {
      -300,  300, -300,  300, -300,  300, -300,  300, -300,  300,
      -300,  300, -300,  300, -300,  300, -300,  300, -300,  300,
      -400,  600
    },
    new int[] {
       500,  476,  405,  294,  155,    0, -155, -294, -405, -476,
      -500, -476, -405, -294, -155,    0,  155,  294,  405,  476,
         0,    0
    },
    new int[] {
         0,  155,  294,  405,  476,  500,  476,  405,  294,  155,
         0, -155, -294, -405, -476, -500, -476, -405, -294, -155,
         0,    0
    },
    new int[] {
      0x00010208, 0x02010310, 0x02030418, 0x04030520, 0x04050628,
      0x06050720, 0x06070818, 0x08070910, 0x08090a08, 0x0a090b00,
      0x0a0b0c08, 0x0c0b0d10, 0x0c0d0e18, 0x0e0d0f20, 0x0e0f1028,
      0x100f1120, 0x10111218, 0x12111310, 0x12130008, 0x00130100,
      0x00021408, 0x02041418, 0x04061428, 0x06081418, 0x080a1408,
      0x0a0c1408, 0x0c0e1418, 0x0e101428, 0x10121418, 0x12001408,
      0x13150100, 0x01150310, 0x03150520, 0x05150720, 0x07150910,
      0x09150b00, 0x0b150d10, 0x0d150f20, 0x0f151120, 0x11151310
    }
  );
  
  /** A Model that looks like a beacon for marking the course. */
  public static final Model BEACON = new Model(
    new int[] {   0,    0,  200,    0, -200,    0},
    new int[] {   0,  200,    0, -200,    0,    0},
    new int[] { 200,    0,    0,    0,    0, -200},
    new int[] {
      0x0001029f, 0x00020307, 0x0003049f, 0x00040107,
      0x01050207, 0x0205039f, 0x03050407, 0x0405019f
    }
  );
  
  /* Private. */
  
  /** A Model that looks like a car body (without wheels). */
  private static class Car extends Model {
    Car(int c) {
      super(
        new int[] {
              0,     0,     0,     0,     0,
           -500,  -750, -1000,  -500,  -250,  -250,
            500,   750,  1000,   500,   250,   250,
              0,  -1000, 1000
        },
        new int[] {
           2500,  1000,   500,     0, -1250,
           2000,  1000,     0, -1000,   250,  -750,
           2000,  1000,     0, -1000,   250,  -750,
           -750, -1250, -1250
        },
        new int[] {
              0,   250,   500,   750,     0,
           -250,     0,     0,  -250,   250,   500,
           -250,     0,     0,  -250,   250,   500,
            500,   560,   560
        },
        new int[] {
          0x00010506+c, 0x05090607+c, 0x06090707+c,
          0x07090806+c, 0x09030a04+c, 0x080a0403+c,
          0x0501092f, 0x01020985, 0x08090a22,
          0x000b0106+c, 0x0b0c0f07+c, 0x0c0d0f07+c,
          0x0d0e0f06+c, 0x0f100304+c, 0x0e041003+c,
          0x0b0f012f, 0x010f0285, 0x0e100f22,
          0x09020f83, 0x0f030902, 0x0a031006+c, 0x10040a0b,
          0x05060701+c, 0x05070800+c,
          0x0b0d0c01+c, 0x0b0e0d00+c,
          0x0b000500+c, 0x08040e00+c, 0x05080b10, 0x0b080e10,
          0x11121300+c, 0x11131207+c
        }
      );
    }
  }
  
  /** A Model that looks like a car from the driver's seat. */
  private static class Cockpit extends Model {
    Cockpit(int c) {
      super(
        new int[] {
              0,     0,     0,     0,     0,     0,
           -500, -1000,  -250,  -250,  -200,  -150, -1000,  -500,
            500,  1000,   250,   250,   200,   150,  1000,   500,
              0,  -300,   300,     0,  -1000,   1000
     },
        new int[] {
           2500,  1000,   500,   250,    50,     0,
           2000,  1000,   750,   250,   250,   125,     0, -1000,
           2000,  1000,   750,   250,   250,   125,     0, -1000,
            750,   500,   500,  -750, -1250, -1250
        },
        new int[] {
              0,   250,     0,   250,   650,   750,
           -250,     0,   250,   250,   250,   500,     0,  -250,
           -250,     0,   250,   250,   250,   500,     0,  -250,
              0,   250,   250,   500,   560,   560
        },
        new int[] {
          0x00010606+c, 0x06080707+c, 0x07081707+c,
          0x07170c07+c, 0x0c170907+c, 0x0c090d06+c,
          0x0601082f, 0x01160883, 0x16020885,
          0x08021784, 0x17020984, 0x09020a83, 0x0a020383,
          0x000e0106+c, 0x0e0f1007+c, 0x100f1807+c,
          0x0f141807+c, 0x18141107+c, 0x14151106+c,
          0x0e10012f, 0x01101683, 0x16100285,
          0x10180284, 0x18110284, 0x11120283, 0x12030283,
          0x090a0b02+c, 0x0b040502+c, 0x05041302+c, 0x13121102+c,
          0x0a030b02, 0x03121302, 0x03130b02, 0x0b130402,
          0x191a1b00+c, 0x191b1a07+c
        }
      );
    }
  }
  
  private int[] xb, yb, zb;
  private int[] faces;
  
  /* Test code. */
  
  public static void main(String[] args) {
    final Lens lens = new Lens(256, 256, 128, 128);
    final SceneImage si = new SceneImage(lens, 512, 512, "Testing Model");
    for (double a=0; ; a+=0.01) {
      final int wx = (int)((1<<16)*Math.cos(a));
      final int wy = (int)((1<<16)*Math.sin(a));
      final Camera c = new Camera(-wx<<9, -wy<<9, 5<<22, wx, wy);
      final Frame f = si.reset(c, null);
      for (int i=0; i<6; i++) {
        final Trajectory t = new Trajectory(0, 0, i<<23);
        final Orientation o = new Orientation(
          1<<14, 0<<14, 0<<14,
          0<<14, 1<<14, 0<<14,
          0<<14, 0<<14, 1<<14
        );
        final Orientation o2 = new Orientation(
          -1<<14, 0<<14, 0<<14,
          0<<14, -1<<14, 0<<14,
          0<<14, 0<<14, 1<<14
        );
        Model.COCKPITS[i].project(o.translate(t, 0, 0, 750), o, f);
        Model.WHEEL.project(o.translate(t, -1200, -1000, 500), o, f);
        Model.WHEEL.project(o.translate(t, -1200, 1500, 500), o, f);
        Model.WHEEL.project(o.translate(t, 1200, -1000, 500), o2, f);
        Model.WHEEL.project(o.translate(t, 1200, 1500, 500), o2, f);
      }
      si.doFrame();
    }
  }
}
