package org.sc3d.apt.jrider.v1;

import java.io.*;
import java.net.*;
import java.util.Random;

 /** The main game loop. */
public class NetworkGame extends Game implements NetworkListener {
  /** Constructs a NetworkGame. This constructor will not return until it has got player numbers from the server for the local players, and sent their configurations. Any key data received during this process will be discarded.
   * @param controller the NetworkController to use. The NetworkController should be newly constructed. This constructor calls many methods of 'controller' during its initialisation, including 'controller.setListener(this)'.
   * @param trackName the name of the track on which to play.
   * @param land the Landscape generated from 'trackName'.
   * @param map a Map of 'land'.
   * @param configs the configurations of the local players. The length of this array defines the number of local players.
   * @param inCar 'true' means a first-person viewpoint, 'false' means third-person.
   * @param fixedCamera If 'inCar', 'true' means a viewpoint that rotates with the Car, 'false' means a viewpoint that looks in the direction of motion. If '!inCar' then 'fixedCamera' is ignored.
   * @param lens the Lens to use for the local players' displays.
   * @param width the width of the windows to open for local players.
   * @param height the height of the windows to open for local players.
   * @throws IOException if something goes wrong with the networking.
   */
  public NetworkGame(
    NetworkController controller,
    String trackName, Landscape land, Map map,
    PlayerConfig[] configs,
    boolean inCar, boolean fixedCamera,
    Lens lens, int width, int height
  ) {
    super(
      controller,
      new SceneImage[6],
      land, map,
      inCar, fixedCamera
    );
    controller.setListener(this);
    this.controller = controller;
    this.lens = lens; this.width = width; this.height = height;
    // Send initialisation messages.
    System.out.println("Initialising...");
    try {
      this.controller.sendExtendedMessage(
        2, // Choose track.
        trackName.getBytes("UTF-8") // Name of track.
      );
    } catch (UnsupportedEncodingException e) {
      e.printStackTrace();
      throw new RuntimeException("No! Withdraw!");
    }
    for (int i=0; i<configs.length; i++) {
      this.controller.sendExtendedMessage(
        3, // Request player number.
        null // No payload.
      );
    }
    // Wait for player numbers to be allocated.
    // The 'processExtendedMessages()' gets called a whole load here and does
    // things like opening windows and negotiating configurations.
    System.out.println("Waiting for player number allocation...");
    this.configs = configs;
    this.playerNumbersStillNeeded = configs.length;
    while (this.playerNumbersStillNeeded>0)
      this.controller.getKeyData(); // Discard key data.
    // Now we have as many player numbers as we can.
    System.out.println("Got all player numbers we're going to get.");
    // Send checksum.
    final int checkSum = land.getCheckSum() + Version.CHECKSUM;
    final byte[] checkSumBytes = new byte[4];
    for (int i=0; i<4; i++) checkSumBytes[i] = (byte)(checkSum >> 8*i);
    this.controller.sendExtendedMessage(
      5, // Send the checksum.
      checkSumBytes
    );
  } 
   
  /* New API. */
  
  /* Implement things in NetworkListener. */
  
  /** Responds to a server-to-client extended message. The behaviour is as follows:<ul>
   * <li>type 0 (sc_chat_message) print to screen.
   * <li>type 1 (sc_track_info) print to screen.
   * <li>type 2 (sc_assign_player) if a player number is needed:<ul>
   *  <li>Call 'NetworkController.setPlayerMapping()'.
   *  <li>Send a 'cs_choose_car_parameters' message to configure the player.
   *  <li>Open a window (we hope this happens before 'start()' is called!).
   *  </ul>
   * <li>type 3 (sc_game_full) print message and abandon players that have not yet been allocated a number.
   * <li>type 4 (sc_car_parameters) call 'this.setPlayerConfig()'.
   * <li>type 5 (sc_time_step) call 'NetworkController.setMSPerWord()'.
   * <li>type 6 (sc_player_leaves) call 'this.setPlayerConfig()'.
   * </ul>
   */
  public void processExtendedMessage(int type, byte[] payload) {
//    System.out.println("processExtendedMessage(");
//    System.out.println("  type="+type);
//    System.out.print("  payload={");
//    if (payload.length>0) {
//      System.out.print(Integer.toHexString(0xFF&payload[0]));
//      for (int i=1; i<payload.length; i++) {
//        System.out.print(" "+Integer.toHexString(0xFF&payload[i]));
//      }
//    }
//    System.out.println("}");
//    System.out.println(")");
    try {
      switch (type) {
        case 0: // Chat message.
          System.out.println("Chat: "+new String(payload, "UTF-8"));
          break;
        case 1: // Track info.
          System.out.println(
            (0xFF&payload[0])+" players on track '"+
            new String(payload, 1, payload.length-1, "UTF-8")
          );
          break;
        case 2: // Allocate player.
          if (this.playerNumbersStillNeeded>0) {
            final int pNum = 0xFF&payload[0];
            final int kNum = --this.playerNumbersStillNeeded;
            // Tell the controller to send key data.
            this.controller.setPlayerMapping(kNum, pNum);
            // Tell the server that the config for 'pNum' is 'configs[kNum]'.
            final byte[] nameBytes = configs[kNum].name.getBytes("UTF-8");
            final byte[] configBytes = new byte[nameBytes.length+2];
            configBytes[0] = (byte)pNum;
            configBytes[1] = (byte)(
              (configs[kNum].fwd ? 0x80 : 0) |
              (configs[kNum].bwd ? 0x40 : 0) |
              (configs[kNum].colour)
            );
            System.arraycopy(nameBytes, 0, configBytes, 2, nameBytes.length);
            this.controller.sendExtendedMessage(
              4, // Set player config.
              configBytes
            );
            // Construct a window for the local player.
            // Hopefully this won't happen after 'start()' is called!
            final SceneImage si =  new SceneImage(
              this.lens,
              this.width, this.height,
              configs[kNum].name
            );
            si.getCanvas().addKeyListener(this.controller);
            this.setSceneImage(pNum, si);
          }
          break;
        case 3: // Game full.
          System.out.println(
            "Game is full; omitting "+this.playerNumbersStillNeeded+" players."
          );
          this.playerNumbersStillNeeded = 0;
          break;
        case 4: // Set car parameters.
          this.setPlayerConfig(
            0xFF&payload[0],
            new PlayerConfig(
              (payload[1]&(1<<7))!=0, // front-wheel drive?
              (payload[1]&(1<<6))!=0, // back-wheel drive?
              0x3F&payload[1], // colour.
              new String(payload, 2, payload.length-2, "UTF-8") // name.
            )
          );
          break;
        case 5: // Set time step.
          final int msPerWord = ((0xFF&payload[0])<<8) | (0xFF&payload[1]);
          this.controller.setMSPerWord(msPerWord);
          break;
        case 6: // Player leaves.
          this.setPlayerConfig(0xFF&payload[0], null);
          break;
        default:
          System.out.println(
            "Unknown server-to-client message, type="+type+
            ", length="+payload.length
          );
      }
    } catch (UnsupportedEncodingException e) {
      e.printStackTrace();
      throw new RuntimeException("No! Withdraw!");
    }
  }
  
  /* Private. */
  
  private int playerNumbersStillNeeded;
  private PlayerConfig[] configs;
  private NetworkController controller;
  private Lens lens;
  private int width, height;
  
  /* Test code. */

  public static void main(String[] args) {
    // Parse arguments.
    if (args.length!=5) throw new IllegalArgumentException(
      "Syntax: java org.sc3d.apt.jrider.v1.Game <host>:<port> <seed> "+
      "<numCars> <camera mode> <drive mode>"
    );
    final int indexOfColon = args[0].indexOf(':');
    if (indexOfColon==-1) throw new IllegalArgumentException(
      "Please supply both the host name and port number of the game server "+
      "not "+args[0]
    );
    final String hostName = args[0].substring(0, indexOfColon);
    final int portNumber = Integer.parseInt(args[0].substring(indexOfColon+1));
    final String trackName = args[1];
    final int numCars = Integer.parseInt(args[2]);
    String cameraMode = args[3].toUpperCase();
    int magnification = 2;
    if (cameraMode.endsWith("-HUGE")) {
      magnification = 4;
      cameraMode = cameraMode.substring(0, cameraMode.length()-5);
    } else if (cameraMode.endsWith("-BIG")) {
      magnification = 3;
      cameraMode = cameraMode.substring(0, cameraMode.length()-4);
    } else if (cameraMode.endsWith("-SMALL")) {
      magnification = 1;
      cameraMode = cameraMode.substring(0, cameraMode.length()-6);
    }
    final boolean fixedCamera = cameraMode.equals("FIXED");
    final boolean motionCamera = cameraMode.equals("MOTION");
    final boolean followCamera = cameraMode.equals("FOLLOW");
    if (!fixedCamera && !motionCamera && !followCamera)
      throw new IllegalArgumentException(
        "Camera mode may be 'fixed', 'motion' or 'follow', with an optional "+
        "suffix '-big' or '-small', but it may not be "+args[3]
      );
    final String driveMode = args[4].toUpperCase();
    final boolean frontWheelDrive = driveMode.equals("FRONT");
    final boolean backWheelDrive =
      driveMode.equals("BACK") || driveMode.equals("REAR");
    final boolean fourWheelDrive = driveMode.equals("FOUR");
    if (!frontWheelDrive && !backWheelDrive && !fourWheelDrive)
      throw new IllegalArgumentException(
        "Drive mode must be 'front', 'rear' or 'four', not "+args[4]
      );
    // Initialise.
    final java.awt.Frame pleaseWait = new java.awt.Frame("Please wait");
    pleaseWait.add(new java.awt.Label("Please wait"));
    pleaseWait.pack();
    pleaseWait.setVisible(true);
    final int res = Math.max(128, 64*magnification);
    final Lens lens = new Lens(2*res, res, res, res);
    final PlayerConfig[] configs = new PlayerConfig[numCars];
    final Random random = new Random();
    for (int i=0; i<numCars; i++) {
      final int colour = random.nextInt(6);
      configs[i] = new PlayerConfig(
        frontWheelDrive | fourWheelDrive, backWheelDrive | fourWheelDrive,
        colour, "Car "+i+" ("+Controller.PLAYERS[colour]+")"
      );
    }
    NetworkController controller;
    try {
      final Socket socket = new Socket(hostName, portNumber);
      socket.setTcpNoDelay(true);
      controller = new NetworkController(
        Controller.DEFAULT_KEYS,
        socket.getInputStream(),
        socket.getOutputStream()
      );
    } catch (UnknownHostException e) {
      throw new IllegalArgumentException("Unknown host: "+hostName);
    } catch (IOException e) {
      e.printStackTrace();
      System.exit(0);
      throw new RuntimeException();
    }
    controller.printKeys(numCars);
    final Landscape land = Landscape.generate(trackName, 11);
    final Map map = new Map(land, 8, "Map of '"+trackName+"'");
    final NetworkGame me = new NetworkGame(
      controller,
      trackName, land, map,
      configs,
      fixedCamera | motionCamera, fixedCamera,
      lens, 256*magnification, 128*magnification
    );
    pleaseWait.setVisible(false);
    me.start();
  }
}
