package org.sc3d.apt.jrider.v1;

/** Represents the instantaneous orientation of a rotating frame of reference, but not its position or velocity. The instantaneous state of the frame is completely specified by an Orientation and a Trajectory together. The Orientation contains the parts that do not change if the origin of the frame is moved. */
public class Orientation {
  /** Constructs an Orientation given values for its fields. */
  public Orientation(
    int xx, int xy, int xz,
    int yx, int yy, int yz,
    int zx, int zy, int zz,
    int wx, int wy, int wz
  ) {
    this.xx = xx; this.xy = xy; this.xz = xz;
    this.yx = yx; this.yy = yy; this.yz = yz;
    this.zx = zx; this.zy = zy; this.zz = zz;
    this.wx = wx; this.wy = wy; this.wz = wz;
  }
  
  /** Constructs an Orientation with zero angular velocity. */
  public Orientation(
    int xx, int xy, int xz,
    int yx, int yy, int yz,
    int zx, int zy, int zz
  ) {
    this.xx = xx; this.xy = xy; this.xz = xz;
    this.yx = yx; this.yy = yy; this.yz = yz;
    this.zx = zx; this.zy = zy; this.zz = zz;
    this.wx = 0; this.wy = 0; this.wz = 0;
  }
  
  /* New API. */
  
  /** The stationary upright Orientation. */
  public static final Orientation ID =
    new Orientation(1<<14, 0, 0, 0, 1<<14, 0, 0, 0, 1<<14);
  
  /** The x-component (north) of a vector of length '1&lt;&lt;14' along the x-axis of the frame. */
  public final int xx;
  
  /** The y-component (east) of the x-axis. */
  public final int xy;
  
  /** The z-component (up) of the x-axis. */
  public final int xz;
  
  /** The x-component of the y-axis. */
  public final int yx;

  /** The y-component of the y-axis. */
  public final int yy;

  /** The z-component of the y-axis. */
  public final int yz;

  /** The x-component of the z-axis. */
  public final int zx;

  /** The y-component of the z-axis. */
  public final int zy;

  /** The z-component of the z-axis. */
  public final int zz;

  /** The x-component (north) of the angular velocity of the frame, in units such that 1/ms is '1&lt;&lt;18' (so 1 radian per second is '1&lt;&lt;8'). The 'w' looks like an omega, the letter conventionally used for angular velocity. */
  public final int wx;

  /** The y-component (east) of the angular velocity of the frame. */
  public final int wy;

  /** The z-component (up) of the angular velocity of the frame. */
  public final int wz;
  
  /** Given the Trajectory of a point in the frame, calculates and returns the Trajectory of another point. The distance between the points should not be more than 16m ('1&lt;&lt;14').
   * @param t the original Trajectory.
   * @param xb the x-component of the new point relative to the old, expressed in the frame of reference, in millimetres, so that the size of the landscape is '1&lt;&lt;20'.
   * @param yb the y-component of the new point relative to the old.
   * @param zb the z-component of the new point relative to the old.
   * @return the new Trajectory.
   */
  public Trajectory translate(Trajectory t, int xb, int yb, int zb) {
    int x = (xb*this.xx + yb*this.yx + zb*this.zx) >> 2;
    int y = (xb*this.xy + yb*this.yy + zb*this.zy) >> 2;
    int z = (xb*this.xz + yb*this.yz + zb*this.zz) >> 2;
    return new Trajectory(
      t.x + x, t.y + y, t.z + z,
      t.vx + ((this.wy*(z>>12) - this.wz*(y>>12)) >> 6),
      t.vy + ((this.wz*(x>>12) - this.wx*(z>>12)) >> 6),
      t.vz + ((this.wx*(y>>12) - this.wy*(x>>12)) >> 6)
    );
  }
}
