package org.sc3d.apt.jrider.v1;

/** Provides an algorithm for generating a random track, given a height field in the form of a LandGen, and a source of random numbers. */
public class Track {
  /** Constructs a Track, choosing the parameters of the cost function and the points that define the course at random. The Track will initially be terrible, which can be fixed by calling its 'improve()' method.
   * @param rand a source of random numbers.
   * @param lg a height field which the Track will try to hug.
   * @param step the log to the base 2 of the spacing of the points, in units such that the size of the landscape is '1&lt;&lt;32'. A good value is about '26' (16m).
   */
  public Track(Random rand, LandGen lg, int step) {
    this(
      lg,
      step,
      1<<24,
      rand.nextUInt(200),
      rand.nextUInt(200),
      rand.nextUInt(200),
      Track.walk(rand, 1<<step, 50+rand.nextUInt(200)),
      rand.nextUInt(200)
    );
  }
  
  /** Generates a Track from the specified LandGen given the parameters of the score function.
   * @param lg a height field which the Track will try to hug.
   * @param step the log to the base 2 of the spacing of the points, in units such that the size of the landscape is '1&lt;&lt;32'. A good value is about '26' (16m).
   * @param danger the height of the smallest precipice that is worth protecting with a barrier, in units such that the size of the landscape is '1&lt;&lt;32'.
   * @param faithfulness the cost (in arbitrary units) of an embankment one step high or a cutting one step deep for one step.
   * @param straightness the cost of a curvature of one radian per step for one step.
   * @param flatness the cost of a gradient of one in one for one step.
   * @param course an array of points through which the Track should pass. These points should be spaced roughly 'step' apart. Unlike all the other parameters, the course may change as the Track is improved.
   * @param gentleness the cost of changing the curvature by one radian per step per step.
   */
  public Track(
    LandGen lg,
    int step,
    int danger,
    int faithfulness,
    int straightness,
    int flatness,
    WayPoint[] course,
    int gentleness
  ) {
    this.lg = lg;
    this.step = step;
    this.danger = danger;
    this.course = course;
    final int scale = Math.max(
      faithfulness, Math.max(
      straightness, Math.max(
      flatness,
      gentleness
    )));
    this.faithfulness = (1<<12)*faithfulness/scale;
    this.straightness = (1<<12)*straightness/(6*scale);
    this.flatness     = (1<<12)*flatness    /(2*scale);
    this.gentleness   = (1<<12)*gentleness  /(20*scale);
//    System.out.println("faithfulness = "+this.faithfulness);
//    System.out.println("straightness = "+this.straightness);
//    System.out.println("flatness     = "+this.flatness);
//    System.out.println("gentleness   = "+this.gentleness);
  }
  
  /* New API. */
  
  /** Represents a point on the course. */
  public static class WayPoint {
    /** The x-coordinate (east) of the point, in units such that the size of the landscape is '1&lt;&lt;32'. */
    public int x;
    /** The y-coordinate (north) of the point, in units such that the size of the landscape is '1&lt;&lt;32'. */
    public int y;
    /** The z-coordinate (up) of the point, in units such that the size of the landscape is '1&lt;&lt;32'. */
    public int z;
  }
  
  /** Generates a circular course, which can be used to initialise the algorithm.
   * @param radius the radius
   * @param length the number of steps in the course.
   */
  public static WayPoint[] circle(final int radius, final int length) {
    WayPoint[] ans = new WayPoint[length];
    for (int i=0; i<length; i++) {
      final WayPoint wp = ans[i] = new WayPoint();
      double a = i*(2*Math.PI/length);
      wp.x = (int)Math.round(radius*Math.cos(a));
      wp.y = (int)Math.round(radius*Math.sin(a));
      wp.z = 0;
    }
    return ans;
  }
  
  /** Generates a random walk course, which can be used to initialise the algorithm. The random walk is closed with a straight line.
   * @param rand a source of random numbers.
   * @param step the step length.
   * @param length the number of steps in the course, excluding the straight line.
   */
  public static WayPoint[] walk(
    final Random rand,
    final int step,
    final int length
  ) {
    // Make a random walk.
    final WayPoint[] wps = new WayPoint[length];
    int x = 1<<31, y = 1<<31;
    double a = rand.nextSInt(710)/113.0;
    for (int i=0; i<length; i++) {
      final WayPoint wp = wps[i] = new WayPoint();
      wp.x = x; wp.y = y; wp.z = 0;
      a += 0.01 * rand.nextSInt(100);
      x += (int)Math.round(step*Math.sin(a));
      y += (int)Math.round(step*Math.cos(a));
    }
    return wps;
  }
  
  /** Modifies the Track slightly so as to improve its score. There's no easy way to decide how many times to call this method; several thousand times is probably about right. */
  public void improve() {
    final int length = this.course.length;
    for (int i=0; i<length; i++) {
      final WayPoint wi = this.course[i];
      final WayPoint wj = this.course[(i+1)%length];
      final WayPoint wk = this.course[(i+2)%length];
      final WayPoint wl = this.course[(i+3)%length];
      // Apply force that corrects the step length.
      final int dx = (wj.x-wi.x) >> 16, dy = (wj.y - wi.y) >> 16;
      final int fl = ((dx*dx+dy*dy) >> (2*this.step-32-14)) - (1<<14);
      if (fl>=(1<<14)) {
        wi.x += dx<<14; wi.y += dy<<14;
        wj.x -= dx<<14; wj.y -= dy<<14;
      } else {
        wi.x += fl*dx; wi.y += fl*dy;
        wj.x -= fl*dx; wj.y -= fl*dy;
        // Apply faithfulness force.
        final int h1 = this.lg.getHeight(wi.x-(1<<26), wi.y);
        final int h2 = this.lg.getHeight(wi.x+(1<<26), wi.y);
        final int h3 = this.lg.getHeight(wi.x, wi.y-(1<<26));
        final int h4 = this.lg.getHeight(wi.x, wi.y+(1<<26));
        final int nx = (h1-h2) >> 19, ny = (h3-h4) >> 19, nz = 1<<8;
        final int dh = ((h1+h2+h3+h4) >> 2) - wi.z;
        final int fh = (dh>>12) * this.faithfulness;
        wi.x += (fh>>8) * nx; wi.y += (fh>>8) * ny; wi.z += (fh>>8) * nz;
        // Apply straightness and gentleness forces.
        final int x3 = (wi.x-3*wj.x+3*wk.x-wl.x) >> 12;
        final int y3 = (wi.y-3*wj.y+3*wk.y-wl.y) >> 12;
        final int z3 = (wi.z-3*wj.z+3*wk.z-wl.z) >> 12;
        final int fsx = this.straightness * x3;
        final int fsy = this.straightness * y3;
        final int fsz = this.straightness * z3;
        wj.x += fsx; wj.y += fsy; wj.z += fsz;
        wk.x -= fsx; wk.y -= fsy; wk.z -= fsz;
        final int fgx = this.gentleness * x3;
        final int fgy = this.gentleness * y3;
        final int fgz = this.gentleness * z3;
        wi.x -=   fgx; wi.y -=   fgy; wi.z -=   fgz;
        wj.x += 3*fgx; wj.y += 3*fgy; wj.z += 3*fgz;
        wk.x -= 3*fgx; wk.y -= 3*fgy; wk.z -= 3*fgz;
        wl.x +=   fgx; wl.y +=   fgy; wl.z +=   fgz;
        // Apply the flatness force.
        final int lx = (wj.x-wi.x) >> (this.step-14);
        final int ly = (wj.y-wi.y) >> (this.step-14);
        final int ff = this.flatness * ((wj.z-wi.z)>>12);
        final int ffl = ((ff>>14) * (ff>>(this.step-14))) >> 14;
        wi.x -= ffl * lx; wi.y -= ffl * ly; wi.z += ff;
        wj.x += ffl * lx; wj.y += ffl * ly; wj.z -= ff;
      }
    }
  }
  
  /** Returns the array of WayPoints that represents the course. */
  public WayPoint[] getCourse() { return this.course; }
  
  /** Returns a an array of LandGen.Cambers containing two Cambers for each WayPoint in the course. The direction of the road at a point is the vector from the previous to the next point, and the sideways gradient is calculated from the curvature. Barriers are placed to defend precipitous drops, especially at the outside of corners.
   * @param tilt the log to the base 2 of the sideways gradient to use on a corner with a curvature of one radian per step. A good value is '-1' (one in two). Max allowable value is '3'.
   * @param width the (full) width of the track in units such that the size of the landscape is '1&lt;&lt;32'.
   */
  public Camber[] getCambers(final int tilt, final int width) {
    // Interpolate WayPoints.
    final int length = 2*this.course.length;
    final WayPoint[] ws = new WayPoint[length];
    WayPoint w1 = this.course[this.course.length-1];
    for (int i=0; i<this.course.length; i++) {
      final WayPoint w2 = this.course[i];
      final WayPoint v1 = ws[2*i] = new WayPoint();
      final WayPoint v2 = ws[2*i+1] = new WayPoint();
      v1.x = w1.x + ((w2.x-w1.x)>>2); v2.x = w2.x + ((w1.x-w2.x)>>2);
      v1.y = w1.y + ((w2.y-w1.y)>>2); v2.y = w2.y + ((w1.y-w2.y)>>2);
      v1.z = w1.z + ((w2.z-w1.z)>>2); v2.z = w2.z + ((w1.z-w2.z)>>2);
      w1 = w2;
    }
    // Construct Cambers.
    Camber[] ans = new Camber[length];
    for (int j=0; j<length; j++) {
      final WayPoint wi = ws[(j+length-1)%length];
      final WayPoint wj = ws[j];
      final WayPoint wk = ws[(j+1)%length];
      // Work out a vector of length about '1<<14' along the road.
      final int dx = (wk.x-wi.x) >> (this.step-14);
      final int dy = (wk.y-wi.y) >> (this.step-14);
      // Work out the acceleration; one radian per step is '1<<14'.
      final int ax = (wi.x-2*wj.x+wk.x) >> (this.step-15);
      final int ay = (wi.y-2*wj.y+wk.y) >> (this.step-15);
      // Construct the Camber.
      final Camber c = ans[j] = new Camber();
      c.x = wj.x; c.y = wj.y; c.z = wj.z;
      c.xx = dy * (width>>15); c.xy = -dx * (width>>15);
      c.xz = (dx*ay-dy*ax) >> (3-tilt);
      c.barrierLeft =
        (c.z-4*c.xz) - this.lg.getHeight(c.x-c.xx, c.y-c.xy) > this.danger;
      c.barrierRight =
        (c.z+4*c.xz) - this.lg.getHeight(c.x+c.xx, c.y+c.xy) > this.danger;
    }
    // Return them.
    return ans;
  }
  
  /* Private. */
  
  private final LandGen lg;
  private final int step;
  private final WayPoint[] course;
  /** A parameter of the score function scaled so that the largest parameter is of the order of '1<<12'. */
  private final int faithfulness, straightness, flatness, gentleness;
  private final int danger;
  
  /* Test code. */
  
  /** Renders a respresentation of this Track and its LandGen on the specified Screen but does not call 'doFrame()'. The screen and landscape should both be 512x512. */
  public void draw(org.sc3d.apt.screen.v4.Screen screen) {
    final int[] pixels = screen.getPixels();
    final int[] heights = this.lg.getHeights();
    for (int i=0; i<512*512; i++)
      pixels[i] = 0x010000 * (0x80 + (0xff & (heights[i^0x3fe00]>>20)));
    for (int i=0; i<this.course.length; i++) {
      final WayPoint wp = this.course[i];
      pixels[(wp.x>>>23) + 512*(~wp.y>>>23)] = 0xffffff;
    }
  }
  
  public static void main(String[] args) throws Exception {
    if (args.length!=1) throw new IllegalArgumentException(
      "Syntax: java org.sc3d.apt.jrider.v1.Track <seed>"
    );
    final LandGen lg = new LandGen(new Random(args[0]+"Land"), 9);
    final Track me = new Track(new Random(args[0]+"Track"), lg, 26);
    final org.sc3d.apt.screen.v4.Screen screen =
      new org.sc3d.apt.screen.v4.Screen(512, 512, "Testing Track");
    while (true) {
      me.draw(screen);
      screen.doFrame();
      for (int i=0; i<100; i++) me.improve();
    }
  }
}
