/*
 * packets.c:
 * Parse and generate packets.
 *
 * Copyright (c) 2003 Chris Lightfoot. All rights reserved.
 * Email: chris@ex-parrot.com; WWW: http://www.ex-parrot.com/~chris/
 *
 */

static const char rcsid[] = "$Id: packets.c,v 1.4 2003/10/19 03:10:23 chris Exp $";

#include <stdbool.h>
#include <stdint.h>

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <netinet/in.h>

#include "protocol.h"
#include "util.h"

/* get_sufficient_buffer LENGTH BUFFER BUFFERLENGTH
 * BUFFER points to at least *BUFFERLENGTH of storage allocated on the heap.
 * If *BUFFERLENGTH is smaller than LENGTH, return a pointer to a reallocated
 * buffer of at least LENGTH bytes, saving its actual size in *BUFFERLENGTH. */
static inline uint8_t *get_sufficient_buffer(const size_t len, uint8_t *buf, size_t *buflen) {
    size_t l;
    if (buf && *buflen >= len) return buf;
    l = *buflen;
    if (!l) ++l;
    while (l < len) l *= 2;
    return xrealloc(buf, *buflen = l);
}

/* scm_to_wire MESSAGE PACKETLENGTH BUFFER BUFFERLENGTH
 * Store in BUFFER, which has been allocated at least *BUFFERLENGTH bytes using
 * malloc(3) the on-the-wire representation of *MESSAGE; return a pointer to
 * BUFFER, which may have been reallocated using realloc(3), saving the number
 * of bytes to transmit in *PACKETLENGTH. If realloc(3) was called, the new
 * number of bytes allocated is saved in *BUFFERLENGTH. In case of error,
 * pktlen is set to zero. XXX really ugly API. */
uint8_t *scm_to_wire(const struct sc_message *msg, size_t *plen, uint8_t *buf, size_t *buflen) {
    assert(msg);
    if (!buf) *buflen = 0;

    if (!msg->scm_extended) {
        /* Copy key press information only. */
        uint32_t u = 0;
        uint8_t p;
        
        buf = get_sufficient_buffer(4, buf, buflen);
        
        for (p = 0; p < MAX_PLAYERS; ++p) {
            enum key_number k;
            
            /* Escape key is not assigned to any particular player. */
            if (msg->scm_player_key_map[p][key_escape])
                u |= 0x80000000;
            
            for (k = 0; k < key_reserved_1; ++k)
                if (msg->scm_player_key_map[p][k]) {
                    u |= (1L << (uint32_t)((int)k + 5 * (int)p));
                }
        }

        u = htonl(u);
        memcpy(buf, &u, 4);
        
        *plen = 4;
    } else {
        uint16_t pktlen, i;
        
        buf = get_sufficient_buffer(4, buf, buflen);

        /* Top byte. */
        buf[0] = 0x40 | (msg->scm_type & 0x3f);
        
        /* Structured packet. */
        switch (msg->scm_type) {
            case sc_chat_message:
            case sc_game_full:
                assert(msg->scm_string);
                pktlen = strlen(msg->scm_string);
                buf = get_sufficient_buffer(3 + pktlen, buf, buflen);
                memcpy(buf + 3, msg->scm_string, pktlen);
                break;

            case sc_track_info:
                assert(msg->scm_string);
                pktlen = 1 + strlen(msg->scm_string);
                buf = get_sufficient_buffer(3 + pktlen, buf, buflen);
                buf[3] = msg->scm_player_num;
                memcpy(buf + 4, msg->scm_string, pktlen - 1);
                break;

            case sc_assign_player:
                pktlen = 1;
                buf = get_sufficient_buffer(3 + pktlen, buf, buflen);
                buf[3] = msg->scm_player_num;
                break;

            case sc_set_car_parameters:
                pktlen = 1 + msg->scm_string_len;
                buf = get_sufficient_buffer(3 + pktlen, buf, buflen);
                buf[3] = msg->scm_player_num;
                memcpy(buf + 4, msg->scm_string, msg->scm_string_len);
                break;

            case sc_time_step:
                pktlen = 2;
                buf = get_sufficient_buffer(3 + pktlen, buf, buflen);
                i = htons(msg->scm_timestep);
                memcpy(buf + 3, &i, 2);
                break;

            case sc_player_leaves:
                pktlen = 1;
                buf = get_sufficient_buffer(3 + pktlen, buf, buflen);
                buf[3] = msg->scm_player_num;
                break;

            default:
                /* Unknown packet type; shouldn't happen. */
                return NULL;
        }
        *plen = 1 + 2 + pktlen;
        pktlen = htons(pktlen);
        memcpy(buf + 1, &pktlen, 2);
    }

    return buf;
}

/* scm_dump STREAM PREFIX MESSAGE
 * Dump information about MESSAGE to STREAM, with each line prefixed by
 * PREFIX. */
void scm_dump(FILE *fp, const char *prefix, const struct sc_message *scm) {
    if (!scm->scm_extended) {
        int i;
        fprintf(fp, "%snormal message\n", prefix);
        for (i = 0; i < MAX_PLAYERS; ++i) {
            int j;
            fprintf(stderr, "%splayer %d:", prefix, i);
            for (j = 0; j < MAX_KEYS; ++j)
                fprintf(stderr, " [%c]", scm->scm_player_key_map[i][j] ? 'X' : ' ');
            fprintf(stderr, "\n");
        }
//        fprintf(fp, "%s: escape %s\n", scm->
    } else {
        const char *type[] = {"sc_chat_message", "sc_track_info", "sc_assign_player", "sc_game_full", "sc_set_car_parameters", "sc_time_step", "sc_player_leaves"};
        fprintf(fp, "%sextended message type %d (%s)\n", prefix, scm->scm_type, type[scm->scm_type]);
        fprintf(fp, "%splayer number %d, time step %d\n", prefix, (int)scm->scm_player_num, (int)scm->scm_timestep);
        if (scm->scm_string)
            fprintf(fp, "%sstring payload: \"%.*s\" (%d bytes)\n", prefix, (int)scm->scm_string_len, scm->scm_string, (int)scm->scm_string_len);
    }
}

/* scm_free MESSAGE
 * Free any storage allocated within MESSAGE (but not MESSAGE itself). */
void scm_free(struct sc_message *msg) {
    xfree(msg->scm_string);
}

/* csm_from_wire MESSAGE BUFFER LENGTH
 * Parse a message received from the client, saving it in *MESSAGE. BUFFER
 * points to at least LENGTH bytes of memory received from the client. On
 * success, fills in the fields of *MESSAGE and returns the number of bytes
 * of BUFFER consumed; if there is not enough data to parse a full packet,
 * return 0; if the on-the-wire data do not represent a valid packet, return
 * a number <0. */
int csm_from_wire(struct cs_message *csm, const uint8_t *buf, const size_t len) {
    struct cs_message csmz = {0};
    assert(csm);
    assert(buf);
    
    if (len == 0)
        return 0;

    *csm = csmz;

//fprintf(stderr, "csm_from_wire %u bytes\n", len);

    if (!(buf[0] & 0x40)) {
        /* Normal key information packet. */
//fprintf(stderr, "  normal packet\n");
        csm->csm_key_up = (buf[0] & 0x80) ? true : false;
//fprintf(stderr, "  key %s\n", csm->csm_key_up ? "UP" : "DOWN");
        csm->csm_key = (buf[0] & 0x7);
//fprintf(stderr, "  key #%d\n", csm->csm_key);
        csm->csm_player_num = (buf[0] >> 3) & 0x7;
//fprintf(stderr, "  player #%d\n", csm->csm_player_num);
        return 1;
    } else {
        uint16_t pktlen;

//fprintf(stderr, "  extended packet\n");
        /* Extended. */
        if (len < 3)
            return 0;
        memcpy(&pktlen, buf + 1, 2);
        pktlen = ntohs(pktlen);
        if (len < 3 + pktlen)
            return 0;

//fprintf(stderr, "  length %d\n", (int)pktlen);
        
        csm->csm_type = buf[0] & 0x3f;
//fprintf(stderr, "  type %d\n", (int)csm->csm_type);

        csm->csm_extended = 1;

        switch (csm->csm_type) {
            /* no payload */
            case cs_list_tracks:
            case cs_claim_player:
                /* NB pktlen *should* be zero, but another value is not an error. */
                return pktlen + 3;

            /* opaque data payload */
            case cs_chat_message:
            case cs_choose_track:
            case cs_send_checksums:
                csm->csm_string = xmalloc(pktlen + 1);
                memcpy(csm->csm_string, buf + 3, pktlen);
                csm->csm_string[pktlen] = 0;
                return pktlen + 3;

            /* structured payload */
            case cs_choose_car_parameters:
                if (pktlen < 2)
                    return -1;
                csm->csm_player_num = buf[3];
                csm->csm_string = xmalloc(csm->csm_string_len = pktlen - 1);
                memcpy(csm->csm_string, buf + 4, csm->csm_string_len);
                return pktlen + 3;
            
            default:
                return -1;
        }
    }
}

void cs_dump(FILE *fp, const char *prefix, struct cs_message *csm) {
    if (!csm->csm_extended) {
        const char *key[] = {"key_left", "key_right", "key_accelerate", "key_brake", "key_change_gear", "key_reserved_1", "key_reserved_2", "key_escape"};
        fprintf(fp, "%snormal message\n", prefix);
        fprintf(fp, "%splayer %d; key motion %s; key %d (%s)\n", prefix, csm->csm_key, csm->csm_key_up ? "UP" : "DOWN", csm->csm_key, key[csm->csm_key]);
    } else {
        const char *type[] = {"cs_chat_message", "cs_list_tracks", "cs_choose_track", "cs_claim_player", "cs_choose_car_parameters", "cs_send_checksums"};
        fprintf(fp, "%sextended message type %d (%s)\n", prefix, csm->csm_type, type[csm->csm_type]);
        fprintf(fp, "%splayer number #%d\n", prefix, csm->csm_player_num);
        if (csm->csm_string)
            fprintf(fp, "%sstring payload: \"%.*s\" (%d bytes)\n", prefix, (int)csm->csm_string_len, csm->csm_string, (int)csm->csm_string_len);
    }
}

/* csm_free MESSAGE
 * Free any storage allocated within MESSAGE (but not MESSAGE itself). */
void csm_free(struct cs_message *csm) {
    xfree(csm->csm_string);
}
