/*
 * server.c:
 * Server object.
 *
 * Copyright (c) 2003 Chris Lightfoot. All rights reserved.
 * Email: chris@ex-parrot.com; WWW: http://www.ex-parrot.com/~chris/
 *
 */

static const char rcsid[] = "$Id: server.c,v 1.3 2003/10/19 03:10:23 chris Exp $";

#include <sys/types.h>
#include <stdbool.h>
#include <stdint.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <netinet/in.h>
#include <sys/socket.h>

#include "game.h"
#include "peer.h"
#include "protocol.h"
#include "server.h"
#include "util.h"

/* struct server
 * Server object internals. */
struct server {
    int s_fd;   /* socket */
    
    peer *s_peers;  /* peers not yet in games. */
    size_t s_num_peers;
    
    game *s_games;  /* games that exist */
    size_t s_num_games;
};

/* server_new ADDRESS
 * Create a new server, listening on ADDRESS. XXX should allow several
 * ADDRESSes, since the server holds all the games. */
server server_new(const struct sockaddr_in *addr) {
    server s;
    int i = 1;
    
    alloc_struct(server, s);
 
    /* Configure socket to listen. SO_REUSEADDR because its absence makes
     * debugging a pain; O_NONBLOCK because accept(2) might block otherwise,
     * which would hang the whole server in an embarrassing manner. */
    if (-1 == (s->s_fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP))
        || -1 == setsockopt(s->s_fd, SOL_SOCKET, SO_REUSEADDR, &i, sizeof i)
        || -1 == bind(s->s_fd, (struct sockaddr*)addr, sizeof *addr)
        || -1 == listen(s->s_fd, SOMAXCONN)
        || -1 == fcntl(s->s_fd, F_SETFL, O_NONBLOCK | fcntl(s->s_fd, F_GETFL)))
        goto fail;

    return s;
fail:
    if (s->s_fd)
        close(s->s_fd);
    xfree(s);
    return NULL;
}

/* server_delete SERVER
 * Delete SERVER and all associated games, disconnecting all the peers. */
void server_delete(server s) {
    int i;
    close(s->s_fd);
    for (i = 0; i < s->s_num_peers; ++i)
        peer_delete(s->s_peers[i]);
    xfree(s->s_peers);
    for (i = 0; i < s->s_num_games; ++i)
        game_delete(s->s_games[i]);
    xfree(s->s_games);
    xfree(s);
}

/* server_pre_select SERVER N READFDS WRITEFDS EXCEPTFDS TIMEOUT
 * Parameters as for select, except N is a pointer. Set up the file descriptor
 * sets as required for server operation. */
void server_pre_select(server s, int *n, fd_set *rd, fd_set *wr, fd_set *ex, struct timeval *tv) {
    int i;
    for (i = 0; i < s->s_num_peers; ++i)
        if (s->s_peers[i]) peer_pre_select(s->s_peers[i], n, rd, wr, ex);
    for (i = 0; i < s->s_num_games; ++i)
        if (s->s_games[i]) game_pre_select(s->s_games[i], n, rd, wr, ex, tv);
    if (s->s_fd + 1 > *n)
        *n = s->s_fd + 1;
    FD_SET(s->s_fd, rd);
}

/* do_list_tracks SERVER PEER
 * Send a list of extant games to PEER. */
static int do_list_tracks(server s, peer p) {
    int i;
    for (i = 0; i < s->s_num_games; ++i) {
        if (s->s_games[i])
            if (!game_send_info(s->s_games[i], p))
                return -1;
    }
    return 0;
}

/* add_peer_to_game SERVER PEER TRACK
 * Add the PEER to the game with the given TRACK name on SERVER, creating
 * such a game if none exists. Returns 1 if the peer was added to a game,
 * 0 if they were not, and -1 if an error occured. */
static int add_peer_to_game(server s, peer p, const char *trackname) {
    int i, ifree = -1;

fprintf(stderr, "add_peer_to_game: peer %s, game %s\n", peer_string(p), trackname);
    
    for (i = 0; i < s->s_num_games; ++i) {
        if (!s->s_games[i])
            ifree = i;
        else if (0 == strcmp(trackname, game_trackname(s->s_games[i]))) {
            int n;
            if ((n = game_add_player(s->s_games[i], p)) >= 0)
                return 1;
            else
                /* Failure to enter a game because it is full is not a protocol error. */
                return (n == -2 ? 0 : -1);
        }
    }

    /* Unable to enter the game. Create a new one. */
    fprintf(stderr, "add_peer_to_game: creating new \"%s\" game for %s\n", trackname, peer_string(p));
    
    if (ifree == -1) {
        /* Allocate more memory for games. */
        size_t max;
        max = s->s_num_games;
        s->s_games = xrealloc(s->s_games, (s->s_num_games = (s->s_num_games ? s->s_num_games * 2 : 8)) * sizeof *s->s_games);
        ifree = (int)max;
        for (; max < s->s_num_games; ++max)
            s->s_games[max] = NULL;
    }
fprintf(stderr, "ZZZ\n");    
    s->s_games[ifree] = game_new(trackname, 0); /* XXX configurable timestep */
    if (game_add_player(s->s_games[ifree], p) >= 0)
        return 1;
    else
        return -1;
}

/* process_client_message SERVER PEER MESSAGE
 * Process a single MESSAGE from a connected PEER on the SERVER, returning
 * 0 if it was processed successfully, -1 if it was not, and 1 if the peer
 * has now been assigned to a game. */
static int q_process_client_message(server s, peer p, struct cs_message *csm) {
    if (!csm->csm_extended)
        return 0;
    switch (csm->csm_type) {
        case cs_chat_message:
            /* Chat message. For the moment, ignore these. */
fprintf(stderr, "process_client_message: ignoring cs_chat_message\n");
            return 0;

        case cs_list_tracks:
fprintf(stderr, "process_client_message: processing cs_list_tracks\n");
            return do_list_tracks(s, p);

        case cs_choose_track:
fprintf(stderr, "process_client_message: processing cs_choose_track\n");
            return add_peer_to_game(s, p, csm->csm_string);

        case cs_claim_player:
        case cs_choose_car_parameters:
        case cs_send_checksums:
            return -1;
    }
    return 0; /* NOTREACHED? */
}

/* receive_packets SERVER PEER
 * Read and process packets from PEER, adding PEER to a game if necessary.
 * Returns 0 on success, -1 in the case of a protocol error which should result
 * in a player's disconnection, or 1 if the peer has been assigned to a game.
 * XXX should eliminate this function and its namesake in game.c, since they do
 * the same thing.... */
static int receive_packets(server s, peer P) {
    uint8_t *buf, *p;
    size_t len;
    int npkts = 0, n, ret = 0;
    p = buf = peer_get_data(P, &len);
    do {
        struct cs_message csm;
        int m;
        
        n = csm_from_wire(&csm, p, len - (p - buf));
        if (n == -1)
            return -1;
        else if (n == 0)
            break;
        
        ++npkts;
        p += n;

        fprintf(stderr, "(server)receive_packets: %s\n", peer_string(P));
        cs_dump(stderr, "(server)receive_packets- ", &csm);
fprintf(stderr, "--> csm type %d\n", csm.csm_type);

        m = q_process_client_message(s, P, &csm);
        csm_free(&csm);
        if (m == -1)
            return -1;
        else if (m == 1) {
            ret = 1;
            break;  /* peer chose game, should leave further messages for the game. */
        }
    } while (n > 0);
    peer_consume(P, p - buf);
    return ret;
}


/* server_post_select SERVER READFDS WRITEFDS EXCEPTFDS
 * Post-select handling for server. Calls into peers and games for their own
 * handling, accepts new connections, and reads packets from peers which are
 * not yet in a game. */
int server_post_select(server s, fd_set *rd, fd_set *wr, fd_set *ex) {
    int i;
    for (i = 0; i < s->s_num_peers; ++i) {
        int n;
        if (s->s_peers[i]) {
            if ((n = peer_post_select(s->s_peers[i], rd, wr, ex)) <= 0) {
                fprintf(stderr, "server_post_select: peer_post_select: %s; disconnecting\n", n == -1 ? strerror(errno) : "dropped connection");
                peer_delete(s->s_peers[i]);
                s->s_peers[i] = NULL;
            } else {
                int m;
                m = receive_packets(s, s->s_peers[i]);
                if (m == -1) {
                    fprintf(stderr, "server_post_select: receive_packets: protocol error\n");
                    peer_delete(s->s_peers[i]);
                    s->s_peers[i] = NULL;
                } else if (m == 1)
                    /* assigned to game */
                    s->s_peers[i] = NULL;
            }
        }
    }
    
    for (i = 0; i < s->s_num_games; ++i)
        if (s->s_games[i]) game_post_select(s->s_games[i], rd, wr, ex);

    /* Handling of incoming connections. */
    if (FD_ISSET(s->s_fd, rd)) {
        int fd;
        if (-1 == (fd = accept(s->s_fd, NULL, NULL))) {
            if (errno != EWOULDBLOCK)
                fprintf(stderr, "server_post_select: accept: %s\n", strerror(errno));
        } else if (-1 == fcntl(fd, F_SETFL, (~O_NONBLOCK) & fcntl(fd, F_GETFL)))
            fprintf(stderr, "server_post_select: fcntl(~O_NONBLOCK): %s\n", strerror(errno));
        else {
            peer p;
            size_t max;
            
            if (!(p = peer_new(fd, NULL))) {
                fprintf(stderr, "server_post_select: peer_new: %s\n", strerror(errno));
            } else {
                for (i = 0; i < s->s_num_peers; ++i)
                    if (!s->s_peers[i]) {
                        s->s_peers[i] = p;
                        p = NULL;
                        break;
                    }

                max = s->s_num_peers;
                s->s_peers = xrealloc(s->s_peers, (s->s_num_peers = (s->s_num_peers ? s->s_num_peers * 2 : 8)) * sizeof *s->s_peers);
                s->s_peers[max++] = p;
                for (; max < s->s_num_peers; ++max)
                    s->s_peers[max] = NULL;
            }
        }
    }

    return 0;
}
