/*
 * util.c:
 * Utility functions.
 *
 * Copyright (c) 2003 Chris Lightfoot. All rights reserved.
 * Email: chris@ex-parrot.com; WWW: http://www.ex-parrot.com/~chris/
 *
 */

static const char rcsid[] = "$Id: util.c,v 1.2 2003/10/19 01:33:03 chris Exp $";

#include <sys/types.h>

#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/time.h>

#include "util.h"

/* xmalloc COUNT
 * Malloc, and abort if malloc fails. */
void *xmalloc(size_t n) {
    void *v;
    v = malloc(n);
    if (!v) abort();
    return v;
}

/* xcalloc NITEMS COUNT
 * As above. */
void *xcalloc(size_t n, size_t m) {
    void *v;
    v = calloc(n, m);
    if (!v) abort();
    return v;
}

/* xrealloc PTR COUNT
 * As above. */
void *xrealloc(void *w, size_t n) {
    void *v;
    v = realloc(w, n);
    if (n != 0 && !v) abort();
    return v;
}

/* xfree PTR
 * Free, ignoring a passed NULL value. */
void xfree(void *v) {
    if (v) free(v);
}

/* xstrdup:
 * Strdup, aborting on failure. */
char *xstrdup(const char *s) {
    char *t;
    t = xmalloc(strlen(s) + 1);
    strcpy(t, s);
    return t;
}


/* compare_timevals TV1 TV2
 * Return -1 if TV1 is before TV2, 1 if TV1 is after TV2, and zero if they
 * represent the same time. */
int compare_timevals(const struct timeval *tv1, const struct timeval *tv2) {
    if (tv1->tv_sec < tv2->tv_sec)
        return -1;
    else if (tv1->tv_sec > tv2->tv_sec)
        return 1;
    else if (tv1->tv_usec < tv2->tv_usec)
        return -1;
    else if (tv1->tv_usec > tv2->tv_usec)
        return 1;
    else
        return 0;
}

/* subtract_timevals TV1 TV2
 * TV1 -= TV2. */
void subtract_timevals(struct timeval *tv1, const struct timeval *tv2) {
    tv1->tv_sec -= tv2->tv_sec;
    tv1->tv_usec -= tv2->tv_usec;
    while (tv1->tv_usec < 0) {
        tv1->tv_usec += 1000000;
        --tv1->tv_sec;
    }
}

/* add_milliseconds_to_timeval TIMEVAL MILLISECONDS
 * Add the given number of MILLISECONDS to the struct TIMEVAL. */
void add_milliseconds_to_timeval(struct timeval *tv, const unsigned int ms) {
    tv->tv_usec += ms * 1000;
    while (tv->tv_usec > 1000000) {
        tv->tv_sec++;
        tv->tv_usec -= 1000000;
    }
}

/* xsignal NUMBER HANDLER
 * Set a signal, with a similar interface to signal(2) using sigaction(2). */
void (*xsignal(int signum, void(*handler)(int)))(int) {
    struct sigaction sa = {0}, sa_old;
    sa.sa_handler = handler;
    sa.sa_flags = SA_RESTART;
    if (sigaction(signum, &sa, &sa_old) == -1)
        return SIG_ERR;
    else
        return sa_old.sa_handler;
}

